<?php

namespace Drupal\autocomplete_extras\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Class AutocompleteExtrasSettingsForm to declare a setting form.
 *
 * @ingroup Content.
 */
class AutocompleteExtrasSettingsForm extends ConfigFormBase {

  /**
   * Returns a unique string identifying the form.
   *
   * The returned ID should be a unique string that can be a valid PHP function
   * name, since it's used in hook implementation names such as
   * hook_form_FORM_ID_alter().
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId(): string {
    return 'autocomplete_extras_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['autocomplete_extras.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('autocomplete_extras.settings');

    $form['menu_link_content'] = [
      '#type' => 'details',
      '#title' => t('Menu Link Content'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['menu_link_content']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable Autocomplete Extras for Menu Link Content'),
      '#default_value' => $config->get('menu_link_content.enabled') ?? FALSE,
    ];

    $form['menu_link_content']['details'] = [
      '#type' => 'fieldset',
      '#states' => [
        'visible' => [
          ':input[name="menu_link_content[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['menu_link_content']['details']['match_limit'] = [
      '#type' => 'number',
      '#title' => t('Number of results'),
      '#min' => 0,
      '#description' => t('The number of suggestions that will be listed. Use <em>0</em> to remove the limit.'),
      '#default_value' => $config->get('menu_link_content.match_limit') ?? NULL,
    ];

    $form['menu_link_content']['details']['min_length'] = [
      '#type' => 'number',
      '#title' => t('Minimum length to trigger autocomplete'),
      '#min' => 1,
      '#default_value' => $config->get('menu_link_content.min_length') ?? NULL,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('autocomplete_extras.settings');

    $values = $form_state->getValues();

    $config->set('menu_link_content.enabled', (bool) $values['menu_link_content']['enabled']);
    if (is_numeric($values['menu_link_content']['details']['match_limit'])) {
      $config->set('menu_link_content.match_limit', (int) $values['menu_link_content']['details']['match_limit']);
    }
    if (is_numeric($values['menu_link_content']['details']['min_length'])) {
      $config->set('menu_link_content.min_length', (int) $values['menu_link_content']['details']['min_length']);
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
