<?php

namespace Drupal\autocomplete_extras\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Field\WidgetInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\link\Plugin\Field\FieldWidget\LinkWidget;

/**
 * Hook implementations for the autocomplete_extras module.
 */
class AutocompleteExtrasHooks {

  use StringTranslationTrait;

  const DEFAULT_MATCH_LIMIT = 10;
  const DEFAULT_MIN_LENGTH = 1;

  /**
   * The class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   A config factory for retrieving required config objects.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_field_widget_complete_form_alter().
   */
  #[Hook('field_widget_complete_form_alter')]
  public function fieldWidgetCompleteFormAlter(array &$field_widget_complete_form, FormStateInterface $form_state, $context): void {
    $widget = $context['widget'];
    if (!($widget instanceof LinkWidget || $widget instanceof EntityReferenceAutocompleteWidget)) {
      return;
    }
    $third_party_settings = $widget->getThirdPartySettings('autocomplete_extras');
    if (empty($third_party_settings)) {
      return;
    }

    if ($widget instanceof LinkWidget) {
      $reference_name = 'uri';
      $match_limit = $third_party_settings['match_limit'];
    }
    else {
      $reference_name = 'target_id';
      $match_limit = $widget->getSetting('match_limit');
    }

    $min_length = $third_party_settings['min_length'];

    if (isset($field_widget_complete_form['widget']['#max_delta'])) {
      $max_delta = $field_widget_complete_form['widget']['#max_delta'];
      for ($i = 0; $i <= $max_delta; $i++) {
        $item = &$field_widget_complete_form['widget'][$i][$reference_name];
        if (empty($item['#selection_settings'])) {
          $item['#selection_settings'] = [];
        }
        $item['#selection_settings']['match_limit'] = $match_limit;
        $item['#attributes']['data-min-length'] = $min_length;
      }
    }
    else {
      $item = &$field_widget_complete_form['widget'][$reference_name];
      if (empty($item['#selection_settings'])) {
        $item['#selection_settings'] = [];
      }
      $item['#selection_settings']['match_limit'] = $match_limit;
      $item['#attributes']['data-min-length'] = $min_length;
    }

    $field_widget_complete_form['#attached']['library'][] = 'autocomplete_extras/autocomplete_extras';
  }

  /**
   * Implements hook_field_widget_third_party_settings_form().
   */
  #[Hook('field_widget_third_party_settings_form')]
  public function fieldWidgetThirdPartySettingsForm(WidgetInterface $widget, $field_definition, $form_mode, array &$form, FormStateInterface $form_state): array {
    if (!($widget instanceof LinkWidget  || $widget instanceof EntityReferenceAutocompleteWidget)) {
      return [];
    }
    $elements = [];
    $third_party_settings = $widget->getThirdPartySettings('autocomplete_extras');
    $settings = $widget->getSettings();
    if (!isset($settings['match_limit'])) {
      $elements['match_limit'] = [
        '#type' => 'number',
        '#title' => $this->t('Number of results'),
        '#description' => $this->t('The number of suggestions that will be listed. Use <em>0</em> to remove the limit.'),
        '#min' => 0,
        '#default_value' => $third_party_settings['match_limit'] ?? self::DEFAULT_MATCH_LIMIT,
      ];
    }

    $elements['min_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum length to trigger autocomplete'),
      '#min' => 1,
      '#default_value' => $third_party_settings['min_length'] ?? self::DEFAULT_MIN_LENGTH,
    ];

    return $elements;
  }

  /**
   * Implements hook_field_widget_settings_summary_alter().
   */
  #[Hook('field_widget_settings_summary_alter')]
  public function fieldWidgetSettingsSummaryAlter(&$summary, $context): void {
    $widget = $context['widget'];
    if (!($widget instanceof LinkWidget || $widget instanceof EntityReferenceAutocompleteWidget)) {
      return;
    }

    $match_limit = $widget->getThirdPartySetting('autocomplete_extras', 'match_limit');
    if (!is_null($match_limit)) {
      $match_limit = ($match_limit > 0) ? $match_limit : $this->t('unlimited');
      $summary[] = $this->t('Autocomplete suggestion list size: @size', ['@size' => $match_limit]);
    }

    $min_length = $widget->getThirdPartySetting('autocomplete_extras', 'min_length');
    if (!empty($min_length)) {
      $summary[] = $this->t('Min length to trigger autocomplete: @setting', ['@setting' => $min_length]);
    }
  }

  /**
   * Implements hook_form_FORM_ID_alter() for menu_link_content_menu_link_content_form.
   */
  #[Hook('form_menu_link_content_menu_link_content_form_alter')]
  public function formMenuLinkContentMenuLinkContentFormAlter(&$form, $form_state): void {
    $config = $this->configFactory->get('autocomplete_extras.settings');

    if ($config->get('menu_link_content.enabled') ?? FALSE) {
      $match_limit = $config->get('menu_link_content.match_limit') ?? self::DEFAULT_MATCH_LIMIT;
      $min_length = $config->get('menu_link_content.min_length') ?? self::DEFAULT_MIN_LENGTH;

      $form['link']['widget'][0]['uri']['#selection_settings']['match_limit'] = $match_limit;
      $form['link']['widget'][0]['uri']['#attributes']['data-min-length'] = $min_length;

      // Add the autocomplete extras library to the form.
      $form['#attached']['library'][] = 'autocomplete_extras/autocomplete_extras';
    }

  }

}
