<?php

declare(strict_types=1);

namespace Drupal\Tests\autocomplete_mixed_matching\FunctionalJavascript;

use PHPUnit\Framework\Attributes\After;

/**
 * Causes a test fail if errors are logged during the test run.
 *
 * With regular Drupal web/functional tests, it can happen that errors occur in
 * the Drupal site, but the test still passes, depending how the assertions are
 * written. Or, even if the test fails, the actual errors are not reported as
 * the reason for the failure.
 *
 * With this trait, such errors are always reported in the test output.
 *
 * The trait can be manually tested like this:
 *   - Insert a failure in a piece of code that is executed during a web
 *     request.
 *     (For this module, it could be a division by zero in the autocomplete
 *     widget.)
 *   - Run a functional test with `->drupalGet()` to a path where that failure
 *     will occur. Also insert a failure in the test method itself, after the
 *     call to `->drupalGet()`.
 *     (For this module, that could be in WidgetTest.)
 *   - Expected: Both failures are reported: The one from the test method, and
 *     the one from the web request.
 */
trait FailOnErrorLogTrait {

  /**
   * Checks for logged errors after a test has run.
   *
   * By intentionally only runs in PhpUnit 10+, because:
   *   - For PhpUnit 9, there is already the overridden ::runTest() method.
   *   - In PhpUnit 9, a failure in a tear down method is not reported if there
   *     is already a failure in the main test method.
   *   - In PhpUnit 9, the order of tear down methods cannot be controlled.
   *     This method must run before the regular tearDown(), otherwise the error
   *     logs would already be deleted at this point.
   *
   * @phpstan-ignore attribute.notFound
   */
  #[After(1)]
  protected function tearDownCheckLoggedErrors(): void {
    $error_log_file = $this->root . '/' . $this->siteDirectory . '/error.log';
    if (file_exists($error_log_file)) {
      $error_log_contents = file_get_contents($error_log_file);
      $this->fail(
        'Errors were logged during the test run:' . "\n" . $error_log_contents,
      );
    }
  }

  /**
   * {@inheritdoc}
   *
   * This method only works in PhpUnit 9, where the parent ::runTest() method is
   * protected. In PhpUnit 10+, it is never called, because the parent method is
   * private.
   */
  protected function runTest(): mixed {
    $result = NULL;
    $failure = NULL;
    try {
      // The method is protected in PhpUnit 9, but private in PhpUnit 10.
      // @phpstan-ignore staticMethod.private
      $result = parent::runTest();
    }
    catch (\Throwable $e) {
      $failure = $e;
    }
    // Check if errors were logged, no matter if the test failed or not.
    $error_log_file = $this->root . '/' . $this->siteDirectory . '/error.log';
    if (file_exists($error_log_file)) {
      $error_log_contents = file_get_contents($error_log_file);
      throw new \Exception(
        'Errors were logged during the test run:' . "\n" . $error_log_contents,
        // Make sure both exceptions are shown in the output.
        previous: $failure,
      );
    }
    if ($failure !== NULL) {
      throw $failure;
    }
    return $result;
  }

}
