<?php

namespace Drupal\autogrid\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Config Task.
 */
class AutogridLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Creates an FieldUiLocalTask object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];
    $config = $this->configFactory->get('autogrid.settings')?->get('entity_types');
    if (empty($config)) {
      return $this->derivatives;
    }

    foreach ($config as $entity_type_id => $value) {
      if ($value == 0) {
        continue;
      }
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      if (!$entity_type) {
        continue;
      }
      $base_entity_type = $entity_type;
      if (method_exists($entity_type, 'getBundleEntityType') && $bundle_type = $entity_type->getBundleEntityType()) {
        // If the entity type uses bundles, use the bundle type instead.
        $base_entity_type = $this->entityTypeManager->getDefinition($bundle_type);
      }
      $base_entity_type_id = $base_entity_type->id();
      // Special handling of Taxonomy. See https://www.drupal.org/node/2822546
      if ($entity_type_id == "taxonomy_vocabulary") {
        $base_route = "entity.{$base_entity_type_id}.overview_form";
      }
      else {
        $base_route = "entity.{$base_entity_type_id}.edit_form";
      }
      $bundle_of = (method_exists($entity_type, 'getBundleOf') && in_array($entity_type->getBundleOf(), $config));
      if (in_array($entity_type_id, $config) || $bundle_of) {
        $this->derivatives["$entity_type_id.autogrid_tab"] = [
          'route_name' => "entity.{$bundle_type}.autogrid",
          'title' => $this->t('Manage data'),
          'base_route' => $base_route,
          'weight' => 90,
        ];
      }
    }

    foreach ($this->derivatives as &$entry) {
      $entry += $base_plugin_definition;
    }

    return $this->derivatives;
  }

}
