<?php

namespace Drupal\autologout_alterable\Hook;

use Drupal\autologout_alterable\AutologoutManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\AccountForm;
use Drupal\user\UserDataInterface;

/**
 * Hook implementations for autologout_alterable.
 */
class AutologoutHooks {

  use StringTranslationTrait;

  /**
   * The autologout configuration.
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a new AutologoutHooks object.
   */
  public function __construct(
    protected AutologoutManagerInterface $autologoutManager,
    protected AccountInterface $currentUser,
    protected ConfigFactoryInterface $configFactory,
    protected UserDataInterface $userData,
  ) {
    $this->config = $configFactory->get('autologout_alterable.settings');
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): ?string {
    $output = '';
    if ($route_name === 'help.page.autologout_alterable') {
      $seconds = $this->autologoutManager->getAutoLogoutProfile()
        ->getSessionExpiresIn();
      $output .= '<h3>' . $this->t('About') . '</h3>';
      $output .= '<p>' . $this->t("This module allows you to force site users to be logged out after a given amount of time due to inactivity after first being presented with a confirmation dialog. Your current logout threshold is %seconds seconds.", ['%seconds' => $seconds]) . '</p>';
    }
    return $output;
  }

  /**
   * Implements hook_page_attachments().
   */
  #[Hook('page_attachments')]
  public function pageAttachments(array &$attachments): void {
    if (!$this->autologoutManager->isEnabled()) {
      return;
    }

    $attachments['#attached']['drupalSettings']['autologout_alterable'] = $this->autologoutManager->getDrupalSettings();
    $attachments['#attached']['library'][] = 'autologout_alterable/autologout';
    $attachments['#cache']['contexts'][] = 'user.roles:authenticated';
    $attachments['#cache']['tags'][] = 'config:autologout_alterable.settings';
  }

  /**
   * Implements hook_user_login().
   */
  #[Hook('user_login')]
  public function userLogin(): void {
    $this->autologoutManager->setLastActivity();
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   *
   * Alter the form "user_form".
   * Adds a field to user to edit that users individual session timeout.
   */
  #[Hook('form_user_form_alter')]
  public function formUserFormAlter(&$form, FormStateInterface $form_state): void {
    $user = NULL;
    $form_object = $form_state->getFormObject();
    if ($form_object instanceof AccountForm && $form_object->getEntity() instanceof AccountInterface) {
      $user = $form_object->getEntity();
    }
    if (!$user) {
      return;
    }
    $user_id = $user->id();
    $access = FALSE;

    // If user-specific thresholds are enabled, and user has access
    // to change, and they are changing their own and only
    // their own timeout, or they are an admin.
    if ($this->config->get('use_individual_logout_threshold') && !$this->currentUser->isAnonymous() && (($this->currentUser->hasPermission('change own autologout_alterable threshold') && $this->currentUser->id() == $user_id) || $this->currentUser->hasPermission('administer autologout_alterable'))) {
      $access = TRUE;

      if ($user_id !== NULL) {
        $timeout = $this->userData->get('autologout_alterable', $user_id, 'session_timeout');
      }
    }

    if ($access) {
      $max_timeout = $this->config->get('max_session_timeout');
      $form['session_timeout_' . $user_id] = [
        '#type' => 'textfield',
        '#title' => t('Your current logout threshold'),
        '#default_value' => $timeout ?? '',
        '#size' => 8,
        '#description' => t('The length of inactivity time, in seconds, before automated log out. Must be between 60 and @max_timeout seconds.', ['@max_timeout' => $max_timeout]),
        '#element_validate' => [[$this, 'userTimeoutValidate']],
      ];

      $form['actions']['submit']['#submit'][] = [$this, 'userProfileSubmit'];
    }
  }

  /**
   * Form validation.
   */
  public function userTimeoutValidate($element, FormStateInterface $form_state): void {
    $max_timeout = $this->config->get('max_session_timeout');
    $timeout = $element['#value'];

    // No value is ok.
    if ($timeout === NULL || $timeout === '') {
      return;
    }

    // Set error if timeout isn't strictly a number between 60 and max.
    if (!is_numeric($timeout) || $timeout < 60 || $timeout > $max_timeout) {
      $form_state->setError($element, t('The timeout must be an integer greater than 60, and less then %max.', ['%max' => $max_timeout]));
    }
  }

  /**
   * Handle submission of timeout threshold in user/edit.
   */
  public function userProfileSubmit(&$form, FormStateInterface $form_state): void {
    $user_id = NULL;
    $form_object = $form_state->getFormObject();
    if ($form_object instanceof AccountForm && $form_object->getEntity() instanceof AccountInterface) {
      $user_id = $form_object->getEntity()->id();
    }
    if (!$user_id) {
      return;
    }

    $access = FALSE;

    // If user-specific thresholds are enabled (the default), and user has
    // access to change, and they are changing their own and only their own
    // timeout, or they are an admin.
    if (!$this->currentUser->isAnonymous() && (($this->currentUser->hasPermission('change own autologout_alterable threshold') && $this->currentUser->id() == $user_id) || $this->currentUser->hasPermission('administer autologout_alterable'))) {
      $access = TRUE;
    }

    // Access is reused here as a security measure. Not only will the element
    // not display but won't submit without access.
    // Do not store config if setting to not store config for every user is
    // TRUE.
    if ($access && $this->config->get('use_individual_logout_threshold')) {
      $timeout = $form_state->getValue('session_timeout_' . $user_id) ??
        $form_state->getValue('session_timeout_') ??
        NULL;
      $this->userData->set('autologout_alterable', $user_id, 'session_timeout', $timeout);
    }
    else {
      $this->userData->set('autologout_alterable', $user_id, 'session_timeout', NULL);
    }
  }

}
