<?php

namespace Drupal\Tests\autologout_alterable\Unit;

use Drupal\autologout_alterable\AutologoutManagerInterface;
use Drupal\autologout_alterable\Hook\AutologoutHooks;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\UserDataInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for AutologoutHooks.
 *
 * @coversDefaultClass \Drupal\autologout_alterable\Hook\AutologoutHooks
 * @group autologout_alterable
 */
class AutologoutHooksTest extends TestCase {

  /**
   * Autologout manager mock.
   */
  private AutologoutManagerInterface&MockObject $autologoutManager;

  /**
   * The mocked current user.
   */
  private AccountInterface&MockObject $currentUser;

  /**
   * The mocked config factory.
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The mocked user data service.
   */
  private UserDataInterface&MockObject $userData;

  /**
   * The mocked immutable config.
   */
  private ImmutableConfig&MockObject $immutableConfig;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->autologoutManager = $this->createMock(AutologoutManagerInterface::class);
    $this->currentUser = $this->createMock(AccountInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->userData = $this->createMock(UserDataInterface::class);
    $this->immutableConfig = $this->createMock(ImmutableConfig::class);

    $this->configFactory
      ->method('get')
      ->with('autologout_alterable.settings')
      ->willReturn($this->immutableConfig);
  }

  /**
   * Helper to get the hooks instance.
   */
  private function getHooks(): AutologoutHooks {
    return new AutologoutHooks(
      $this->autologoutManager,
      $this->currentUser,
      $this->configFactory,
      $this->userData,
    );
  }

  /**
   * @covers ::pageAttachments
   */
  public function testPageAttachmentsDisabled(): void {
    $this->autologoutManager
      ->method('isEnabled')
      ->willReturn(FALSE);

    $attachments = [];
    $this->getHooks()->pageAttachments($attachments);

    $this->assertSame([], $attachments, 'Attachments should remain empty when autologout is disabled.');
  }

  /**
   * @covers ::pageAttachments
   */
  public function testPageAttachmentsEnabled(): void {
    $this->autologoutManager
      ->method('isEnabled')
      ->willReturn(TRUE);

    $settings = ['foo' => 'bar'];
    $this->autologoutManager
      ->method('getDrupalSettings')
      ->willReturn($settings);

    $attachments = [];
    $this->getHooks()->pageAttachments($attachments);

    $this->assertArrayHasKey('#attached', $attachments);
    $this->assertArrayHasKey('drupalSettings', $attachments['#attached']);
    $this->assertArrayHasKey('autologout_alterable', $attachments['#attached']['drupalSettings']);
    $this->assertSame($settings, $attachments['#attached']['drupalSettings']['autologout_alterable']);

    $this->assertArrayHasKey('library', $attachments['#attached']);
    $this->assertContains('autologout_alterable/autologout', $attachments['#attached']['library']);

    $this->assertArrayHasKey('#cache', $attachments);
    $this->assertContains('user.roles:authenticated', $attachments['#cache']['contexts']);
    $this->assertContains('config:autologout_alterable.settings', $attachments['#cache']['tags']);
  }

  /**
   * @covers ::userLogin
   */
  public function testUserLogin(): void {
    $this->autologoutManager
      ->expects($this->once())
      ->method('setLastActivity');

    $this->getHooks()->userLogin();
  }

}
