<?php

namespace Drupal\Tests\autologout_alterable\Unit;

use Drupal\autologout_alterable\AutologoutManagerInterface;
use Drupal\autologout_alterable\Hook\AutologoutHooks;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Select;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\user\UserDataInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for AutologoutHooks.
 *
 * @coversDefaultClass \Drupal\autologout_alterable\Hook\AutologoutHooks
 * @group autologout_alterable
 */
class AutologoutHooksTest extends TestCase {

  /**
   * Autologout manager mock.
   */
  private AutologoutManagerInterface&MockObject $autologoutManager;

  /**
   * The mocked current user.
   */
  private AccountInterface&MockObject $currentUser;

  /**
   * The mocked config factory.
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The mocked user data service.
   */
  private UserDataInterface&MockObject $userData;

  /**
   * The mocked immutable config.
   */
  private ImmutableConfig&MockObject $immutableConfig;

  /**
   * The mocked database connection.
   */
  private Connection&MockObject $database;

  /**
   * The mocked state service.
   */
  private StateInterface&MockObject $state;

  /**
   * The mocked queue factory.
   */
  private QueueFactory&MockObject $queueFactory;

  /**
   * The mocked time service.
   */
  private TimeInterface&MockObject $time;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->autologoutManager = $this->createMock(AutologoutManagerInterface::class);
    $this->currentUser = $this->createMock(AccountInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->userData = $this->createMock(UserDataInterface::class);
    $this->immutableConfig = $this->createMock(ImmutableConfig::class);
    $this->database = $this->createMock(Connection::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->queueFactory = $this->createMock(QueueFactory::class);
    $this->time = $this->createMock(TimeInterface::class);

    $this->configFactory
      ->method('get')
      ->with('autologout_alterable.settings')
      ->willReturn($this->immutableConfig);
  }

  /**
   * Helper to get the hooks instance.
   */
  private function getHooks(): AutologoutHooks {
    return new AutologoutHooks(
      $this->autologoutManager,
      $this->currentUser,
      $this->configFactory,
      $this->userData,
      $this->database,
      $this->state,
      $this->queueFactory,
      $this->time,
    );
  }

  /**
   * @covers ::pageAttachments
   */
  public function testPageAttachmentsDisabled(): void {
    $this->autologoutManager
      ->method('isEnabled')
      ->willReturn(FALSE);

    $attachments = [];
    $this->getHooks()->pageAttachments($attachments);

    $this->assertSame([], $attachments, 'Attachments should remain empty when autologout is disabled.');
  }

  /**
   * @covers ::pageAttachments
   */
  public function testPageAttachmentsEnabled(): void {
    $this->autologoutManager
      ->method('isEnabled')
      ->willReturn(TRUE);

    $settings = ['foo' => 'bar'];
    $this->autologoutManager
      ->method('getDrupalSettings')
      ->willReturn($settings);

    $attachments = [];
    $this->getHooks()->pageAttachments($attachments);

    $this->assertArrayHasKey('#attached', $attachments);
    $this->assertArrayHasKey('drupalSettings', $attachments['#attached']);
    $this->assertArrayHasKey('autologout_alterable', $attachments['#attached']['drupalSettings']);
    $this->assertSame($settings, $attachments['#attached']['drupalSettings']['autologout_alterable']);

    $this->assertArrayHasKey('library', $attachments['#attached']);
    $this->assertContains('autologout_alterable/autologout', $attachments['#attached']['library']);

    $this->assertArrayHasKey('#cache', $attachments);
    $this->assertContains('user.roles:authenticated', $attachments['#cache']['contexts']);
    $this->assertContains('config:autologout_alterable.settings', $attachments['#cache']['tags']);
  }

  /**
   * @covers ::userLogin
   */
  public function testUserLogin(): void {
    $this->autologoutManager
      ->expects($this->once())
      ->method('setLastActivity');

    $this->getHooks()->userLogin();
  }

  /**
   * @covers ::cron
   */
  public function testCronDisabledWhenUseCronIsFalse(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', FALSE],
        ['enabled', TRUE],
      ]);

    $this->state
      ->expects($this->never())
      ->method('get');

    $this->getHooks()->cron();
  }

  /**
   * @covers ::cron
   */
  public function testCronDisabledWhenEnabledIsFalse(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', FALSE],
      ]);

    $this->state
      ->expects($this->never())
      ->method('get');

    $this->getHooks()->cron();
  }

  /**
   * @covers ::cron
   */
  public function testCronProcessesSessionsAndAddsToQueue(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $this->state
      ->expects($this->atLeastOnce())
      ->method('get')
      ->with('autologout_alterable.pending_session_ids', [])
      ->willReturn([]);

    $this->time
      ->method('getRequestTime')
      ->willReturn(3600);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchCol')->willReturn(['session1', 'session2']);
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $queue = $this->createMock(QueueInterface::class);
    $queue
      ->expects($this->exactly(2))
      ->method('createItem')
      ->with($this->callback(function ($item) {
        return isset($item['sid']) && in_array($item['sid'], ['session1', 'session2']);
      }));

    $this->queueFactory
      ->method('get')
      ->with('autologout_alterable_session_check')
      ->willReturn($queue);

    $this->state
      ->expects($this->once())
      ->method('set')
      ->with(
        'autologout_alterable.pending_session_ids',
        ['session1' => TRUE, 'session2' => TRUE]
      );

    $this->getHooks()->cron();
  }

  /**
   * @covers ::cron
   */
  public function testCronCleansUpStateWhenOver500Items(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $pending_sessions = array_fill_keys(array_map(fn($i) => "session{$i}", range(1, 501)), TRUE);

    $this->state
      ->method('get')
      ->with('autologout_alterable.pending_session_ids', [])
      ->willReturn($pending_sessions);

    $this->time
      ->method('getRequestTime')
      ->willReturn(3600);

    $statement1 = $this->createMock(StatementInterface::class);
    $statement1->method('fetchCol')->willReturn(array_map(fn($i) => "session{$i}", range(1, 50)));

    $statement2 = $this->createMock(StatementInterface::class);
    $statement2->method('fetchCol')->willReturn(['new_session1', 'new_session2']);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturnOnConsecutiveCalls($statement1, $statement2);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $queue = $this->createMock(QueueInterface::class);
    $queue
      ->expects($this->exactly(2))
      ->method('createItem');

    $this->queueFactory
      ->method('get')
      ->with('autologout_alterable_session_check')
      ->willReturn($queue);

    $this->state
      ->expects($this->exactly(2))
      ->method('set')
      ->with('autologout_alterable.pending_session_ids', $this->anything());

    $this->getHooks()->cron();
  }

}
