<?php

namespace Drupal\Tests\autologout_alterable\Unit;

use Drupal\autologout_alterable\AutologoutManagerInterface;
use Drupal\autologout_alterable\Events\AutologoutCronProfileAlterEvent;
use Drupal\autologout_alterable\Plugin\QueueWorker\AutologoutSessionCheckWorker;
use Drupal\autologout_alterable\Utility\AutologoutProfileInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Delete;
use Drupal\Core\Database\Query\Select;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Unit tests for AutologoutSessionCheckWorker.
 *
 * @coversDefaultClass \Drupal\autologout_alterable\Plugin\QueueWorker\AutologoutSessionCheckWorker
 * @group autologout_alterable
 */
class AutologoutSessionCheckWorkerTest extends TestCase {

  /**
   * The mocked database connection.
   */
  private Connection&MockObject $database;

  /**
   * The mocked entity type manager.
   */
  private EntityTypeManagerInterface&MockObject $entityTypeManager;

  /**
   * The mocked state service.
   */
  private StateInterface&MockObject $state;

  /**
   * The mocked session manager.
   */
  private SessionManagerInterface&MockObject $sessionManager;

  /**
   * The mocked config factory.
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The mocked account switcher.
   */
  private AccountSwitcherInterface&MockObject $accountSwitcher;

  /**
   * The mocked autologout manager.
   */
  private AutologoutManagerInterface&MockObject $autologoutManager;

  /**
   * The mocked event dispatcher.
   */
  private EventDispatcherInterface&MockObject $eventDispatcher;

  /**
   * The mocked time service.
   */
  private TimeInterface&MockObject $time;

  /**
   * The mocked logger channel factory.
   */
  private LoggerChannelFactoryInterface&MockObject $loggerFactory;

  /**
   * The mocked logger channel.
   */
  private LoggerChannelInterface&MockObject $logger;

  /**
   * The mocked immutable config.
   */
  private ImmutableConfig&MockObject $immutableConfig;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->sessionManager = $this->createMock(SessionManagerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->accountSwitcher = $this->createMock(AccountSwitcherInterface::class);
    $this->autologoutManager = $this->createMock(AutologoutManagerInterface::class);
    $this->eventDispatcher = $this->createMock(EventDispatcherInterface::class);
    $this->time = $this->createMock(TimeInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->immutableConfig = $this->createMock(ImmutableConfig::class);

    $this->loggerFactory
      ->method('get')
      ->with('autologout_alterable')
      ->willReturn($this->logger);

    $this->configFactory
      ->method('get')
      ->with('autologout_alterable.settings')
      ->willReturn($this->immutableConfig);
  }

  /**
   * Helper to get the worker instance.
   */
  private function getWorker(): AutologoutSessionCheckWorker {
    return new AutologoutSessionCheckWorker(
      [],
      'autologout_alterable_session_check',
      [],
      $this->database,
      $this->entityTypeManager,
      $this->state,
      $this->sessionManager,
      $this->configFactory,
      $this->accountSwitcher,
      $this->autologoutManager,
      $this->eventDispatcher,
      $this->time,
      $this->loggerFactory,
    );
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemDisabledWhenUseCronIsFalse(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', FALSE],
        ['enabled', TRUE],
      ]);

    $this->database
      ->expects($this->never())
      ->method('select');

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemDisabledWhenEnabledIsFalse(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', FALSE],
      ]);

    $this->database
      ->expects($this->never())
      ->method('select');

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemReturnsEarlyWhenNoSid(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $this->database
      ->expects($this->never())
      ->method('select');

    $this->getWorker()->processItem([]);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemReturnsEarlyWhenDataIsNotArray(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $this->database
      ->expects($this->never())
      ->method('select');

    $this->getWorker()->processItem('invalid_data');
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemRemovesSessionFromStateWhenSessionNotFound(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn(FALSE);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $this->state
      ->expects($this->once())
      ->method('get')
      ->with('autologout_alterable.pending_session_ids', [])
      ->willReturn(['test_session' => TRUE]);

    $this->state
      ->expects($this->once())
      ->method('set')
      ->with('autologout_alterable.pending_session_ids', []);

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemReturnsEarlyWhenUidIsInvalid(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn([
      'uid' => 0,
      'session' => 'test_session_data',
    ]);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $this->state
      ->expects($this->never())
      ->method('set');

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::decodeAutologoutSessionData
   */
  public function testDecodeAutologoutSessionDataReturnsFalseOnInvalidData(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn([
      'uid' => 1,
      'session' => 'invalid_session_data',
    ]);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $user = $this->createMock(UserInterface::class);
    $userStorage = $this->createMock(EntityStorageInterface::class);
    $userStorage
      ->method('load')
      ->with(1)
      ->willReturn($user);

    $this->entityTypeManager
      ->method('getStorage')
      ->with('user')
      ->willReturn($userStorage);

    $this->logger
      ->expects($this->once())
      ->method('error')
      ->with('Unable to unserialize session skipping.');

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::removeSessionFromState
   */
  public function testRemoveSessionFromState(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn(FALSE);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $this->state
      ->method('get')
      ->with('autologout_alterable.pending_session_ids', [])
      ->willReturn([
        'test_session' => TRUE,
        'other_session' => TRUE,
      ]);

    $this->state
      ->expects($this->once())
      ->method('set')
      ->with('autologout_alterable.pending_session_ids', ['other_session' => TRUE]);

    $this->getWorker()->processItem(['sid' => 'test_session']);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemReturnsEarlyWhenTimeoutNotApplicable(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    // Valid session string with properly serialized data.
    $session_array = [
      'autologout_alterable_session_start' => 1000000,
      'autologout_alterable_last_activity' => 1000100,
      'autologout_alterable_profile_id' => 'default',
    ];
    $serialized_session = serialize($session_array);
    $session_data_string = "_sf2_attributes|{$serialized_session}_sf2_meta|";

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn([
      'uid' => 1,
      'session' => $session_data_string,
    ]);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $user = $this->createMock(UserInterface::class);
    $userStorage = $this->createMock(EntityStorageInterface::class);
    $userStorage
      ->expects($this->once())
      ->method('load')
      ->with(1)
      ->willReturn($user);

    $this->entityTypeManager
      ->expects($this->once())
      ->method('getStorage')
      ->with('user')
      ->willReturn($userStorage);

    $this->time
      ->method('getCurrentTime')
      ->willReturn(1000000);

    $this->autologoutManager
      ->expects($this->once())
      ->method('getDefaultTimeout')
      ->with($user)
      ->willReturn(AutologoutProfileInterface::EXPIRES_IN_NOT_APPLICABLE);

    $this->accountSwitcher
      ->expects($this->never())
      ->method('switchTo');

    $this->accountSwitcher
      ->expects($this->never())
      ->method('switchBack');

    $this->getWorker()->processItem(['sid' => 'test_session']);

    // Assertion to ensure the test completes successfully without exceptions.
    $this->assertTrue(TRUE);
  }

  /**
   * @covers ::processItem
   */
  public function testProcessItemLoadsUserSuccessfully(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    // Valid session string with properly serialized data.
    $session_array = [
      'autologout_alterable_session_start' => 1000000,
      'autologout_alterable_last_activity' => 1000100,
      'autologout_alterable_profile_id' => 'default',
    ];
    $serialized_session = serialize($session_array);
    $session_data_string = "_sf2_attributes|{$serialized_session}_sf2_meta|";

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn([
      'uid' => 1,
      'session' => $session_data_string,
    ]);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $user = $this->createMock(UserInterface::class);
    $userStorage = $this->createMock(EntityStorageInterface::class);
    $userStorage
      ->expects($this->once())
      ->method('load')
      ->with(1)
      ->willReturn($user);

    $this->entityTypeManager
      ->expects($this->once())
      ->method('getStorage')
      ->with('user')
      ->willReturn($userStorage);

    $this->time
      ->method('getCurrentTime')
      ->willReturn(1000000);

    $this->autologoutManager
      ->method('getDefaultTimeout')
      ->with($user)
      ->willReturn(1800);

    $this->autologoutManager
      ->method('calculateDefaultSessionExpiration')
      ->willReturn(new \DateTime('@1003700'));

    $redirectUrl = $this->createMock(Url::class);
    $this->autologoutManager
      ->method('getDefaultRedirectUrl')
      ->willReturn($redirectUrl);

    $event = $this->createMock(AutologoutCronProfileAlterEvent::class);
    $event->method('preventSessionDelete')->willReturn(FALSE);
    $event->method('getAutologoutProfile')->willReturnCallback(function () {
      $profile = $this->createMock(AutologoutProfileInterface::class);
      $profile->method('getSessionExpiresIn')->willReturn(0);
      return $profile;
    });

    $this->eventDispatcher
      ->method('dispatch')
      ->willReturn($event);

    $deleteQuery = $this->createMock(Delete::class);
    $deleteQuery->method('condition')->willReturnSelf();
    $deleteQuery->method('execute');

    $this->database
      ->method('delete')
      ->with('sessions')
      ->willReturn($deleteQuery);

    $this->state
      ->method('get')
      ->willReturn(['test_session' => TRUE]);

    $this->state
      ->method('set');

    $this->accountSwitcher
      ->expects($this->once())
      ->method('switchTo')
      ->with($user);

    $this->accountSwitcher
      ->expects($this->once())
      ->method('switchBack');

    $this->getWorker()->processItem(['sid' => 'test_session']);

    // Assertion to ensure the test completes successfully without exceptions.
    $this->assertTrue(TRUE);
  }

  /**
   * @covers ::processItem
   * @covers ::deleteSession
   */
  public function testProcessItemDeletesSessionWhenUserNotFound(): void {
    $this->immutableConfig
      ->method('get')
      ->willReturnMap([
        ['use_cron', TRUE],
        ['enabled', TRUE],
      ]);

    // Valid session string with properly serialized data.
    $session_array = [
      'autologout_alterable_session_start' => 1000000,
      'autologout_alterable_last_activity' => 1000100,
      'autologout_alterable_profile_id' => 'default',
    ];
    $serialized_session = serialize($session_array);
    $session_data_string = "_sf2_attributes|{$serialized_session}_sf2_meta|";

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAssoc')->willReturn([
      'uid' => 999,
      'session' => $session_data_string,
    ]);

    $query = $this->createMock(Select::class);
    $query->method('fields')->willReturnSelf();
    $query->method('condition')->willReturnSelf();
    $query->method('execute')->willReturn($statement);

    $deleteQuery = $this->createMock(Delete::class);
    $deleteQuery->method('condition')->willReturnSelf();
    $deleteQuery->expects($this->once())->method('execute');

    $this->database
      ->method('select')
      ->with('sessions', 's')
      ->willReturn($query);

    $this->database
      ->expects($this->once())
      ->method('delete')
      ->with('sessions')
      ->willReturn($deleteQuery);

    $userStorage = $this->createMock(EntityStorageInterface::class);
    $userStorage
      ->expects($this->once())
      ->method('load')
      ->with(999)
      ->willReturn(NULL);

    $this->entityTypeManager
      ->expects($this->once())
      ->method('getStorage')
      ->with('user')
      ->willReturn($userStorage);

    $this->time
      ->method('getCurrentTime')
      ->willReturn(1000000);

    $this->accountSwitcher
      ->expects($this->never())
      ->method('switchTo');

    $this->accountSwitcher
      ->expects($this->never())
      ->method('switchBack');

    $this->getWorker()->processItem(['sid' => 'test_session']);

    // Assertion to ensure session deletion was properly verified.
    $this->assertTrue(TRUE);
  }

}
