<?php

namespace Drupal\autologout_alterable\Plugin\QueueWorker;

use Drupal\autologout_alterable\AutologoutManagerInterface;
use Drupal\autologout_alterable\Events\AutologoutCronProfileAlterEvent;
use Drupal\autologout_alterable\Events\AutologoutEvents;
use Drupal\autologout_alterable\Utility\AutologoutProfile;
use Drupal\autologout_alterable\Utility\AutologoutProfileInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Processes autologout session checks.
 */
#[QueueWorker(
  id: 'autologout_alterable_session_check',
  title: new TranslatableMarkup('Autologout Session Check Worker'),
  cron: ['time' => 60]
)]
class AutologoutSessionCheckWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The autologout config.
   */
  protected ImmutableConfig $autoLogoutConfig;

  /**
   * The autologout logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new AutologoutSessionCheckWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Session\SessionManagerInterface $sessionManager
   *   The session manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountSwitcherInterface $accountSwitcher
   *   The account switcher.
   * @param \Drupal\autologout_alterable\AutologoutManagerInterface $autologoutManager
   *   The autologout manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $channelFactory
   *   The logger channel factory.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected Connection $database,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected StateInterface $state,
    protected SessionManagerInterface $sessionManager,
    protected ConfigFactoryInterface $configFactory,
    protected AccountSwitcherInterface $accountSwitcher,
    protected AutologoutManagerInterface $autologoutManager,
    protected EventDispatcherInterface $eventDispatcher,
    protected TimeInterface $time,
    LoggerChannelFactoryInterface $channelFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->autoLogoutConfig = $this->configFactory->get('autologout_alterable.settings');
    $this->logger = $channelFactory->get('autologout_alterable');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('state'),
      $container->get('session_manager'),
      $container->get('config.factory'),
      $container->get('account_switcher'),
      $container->get('autologout_alterable.manager'),
      $container->get('event_dispatcher'),
      $container->get('datetime.time'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    if (!$this->autoLogoutConfig->get('use_cron') || !$this->autoLogoutConfig->get('enabled')) {
      return;
    }
    $sid = is_array($data) ? $data['sid'] ?? NULL : NULL;
    if (!$sid) {
      return;
    }

    $has_switched_account = FALSE;
    try {
      $session_data = $this->database->select('sessions', 's')
        ->fields('s', ['uid', 'session'])
        ->condition('sid', $sid)
        ->execute()
        ->fetchAssoc();

      if (!$session_data || !is_array($session_data)) {
        $this->removeSessionFromState($sid);
        return;
      }

      $uid = $session_data['uid'] ?? NULL;

      if (!$uid || $uid < 1) {
        // Keep session in state so we don't process it again.
        return;
      }

      /** @var \Drupal\user\UserInterface|null $user */
      $user = $this->entityTypeManager->getStorage('user')->load($uid);
      if (!$user) {
        // Delete session if user no longer exists.
        $this->deleteSession($sid);
        return;
      }

      $session = $session_data['session'] ?? NULL;

      if (!$session || !is_string($session)) {
        // Keep the session in state so we don't process it again.
        return;
      }

      $session = $this->decodeAutologoutSessionData($session);
      if ($session === FALSE) {
        $this->logger->error('Unable to unserialize session skipping.');
        // Keep the session in state so we don't process it again.
        return;
      }

      if (!$session['autologout_alterable_profile_id']) {
        // This session does not have a profile, skip it.
        return;
      }

      $default_timeout = $this->autologoutManager->getDefaultTimeout($user);
      if ($default_timeout === AutologoutProfileInterface::EXPIRES_IN_NOT_APPLICABLE) {
        // This user should not be handled by autologout.
        return;
      }

      $this->accountSwitcher->switchTo($user);
      $has_switched_account = TRUE;

      $session_start = new \DateTime('@' . $session['autologout_alterable_session_start']);
      $last_activity = new \DateTime('@' . $session['autologout_alterable_last_activity']);
      $extendable = TRUE;
      $session_expiration = $this->autologoutManager->calculateDefaultSessionExpiration($session_start, $last_activity, $default_timeout, $extendable);
      $redirect_url = $this->autologoutManager->getDefaultRedirectUrl();

      $profile = new AutologoutProfile(
        $session_start,
        $last_activity,
        $session_expiration,
        $redirect_url,
        $extendable,
        $session['autologout_alterable_profile_id'],
      );

      $event = new AutologoutCronProfileAlterEvent($profile, $user);
      $this->eventDispatcher->dispatch($event, AutologoutEvents::AUTOLOGOUT_CRON_PROFILE_ALTER_CRON);

      if ($event->preventSessionDelete()) {
        // Keep session in state so we don't process it again.
        return;
      }

      $profile = $event->getAutologoutProfile();

      if ($profile->getSessionExpiresIn() === AutologoutProfileInterface::EXPIRES_IN_NOT_APPLICABLE) {
        // Keep session in state so we don't process it again.
        return;
      }

      if ($profile->getSessionExpiresIn() <= 0) {
        // Delete expired session.
        $this->deleteSession($sid);
      }

      // Remove the session from state after processing so we can process it
      // again the next time the cron runs.
      $this->removeSessionFromState($sid);
      return;
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing autologout session check: @message', ['@message' => $e->getMessage()]);
    }
    finally {
      if ($has_switched_account) {
        try {
          $this->accountSwitcher->switchBack();
        }
        catch (\Exception $e) {
          // Noop.
        }
      }
    }
  }

  /**
   * Decode the autologout session data from the session string.
   *
   * @param string $session_string
   *   The session string to decode.
   *
   * @return array|false
   *   The decoded session data or FALSE if decode fails.
   */
  protected function decodeAutologoutSessionData(string $session_string): array|false {
    $regex = '/_sf2_attributes\|(a:\d+:{.*})_\w+\|/sU';

    // Attempt to match the pattern and capture the serialized string.
    if (preg_match($regex, $session_string, $matches)) {
      $serialized_attributes = $matches[1];
      $decoded_attributes = unserialize($serialized_attributes);
      if (!$decoded_attributes || !is_array($decoded_attributes)) {
        return FALSE;
      }
      return [
        'autologout_alterable_session_start' => $decoded_attributes['autologout_alterable_session_start'] ?? $this->time->getCurrentTime(),
        'autologout_alterable_last_activity' => $decoded_attributes['autologout_alterable_last_activity'] ?? $this->time->getCurrentTime(),
        'autologout_alterable_profile_id' => $decoded_attributes['autologout_alterable_profile_id'] ?? NULL,
      ];
    }

    return FALSE;
  }

  /**
   * Delete a session from the database.
   *
   * @param string $sid
   *   The session ID to delete.
   */
  protected function deleteSession(string $sid): void {
    $this->database->delete('sessions')
      ->condition('sid', $sid)
      ->execute();
  }

  /**
   * Remove a session ID from the state.
   *
   * @param string $sid
   *   The session ID to remove.
   */
  protected function removeSessionFromState(string $sid): void {
    $state_key = 'autologout_alterable.pending_session_ids';
    $pending_session_ids = $this->state->get($state_key, []);
    unset($pending_session_ids[$sid]);
    $this->state->set($state_key, $pending_session_ids);
  }

}
