<?php

namespace Drupal\automatic_site_icon_picker\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\file\FileRepository;
use Drupal\Core\Http\ClientFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Formatter to display a link field with a favicon icon.
 *
 * Downloads and caches favicons from a remote API and displays them
 * next to link field output.
 *
 * @FieldFormatter(
 *   id = "automatic_site_icon_picker_link",
 *   label = @Translation("Link (favicon)"),
 *   field_types = {
 *     "link"
 *   }
 * )
 */
class AutomaticSiteIconPickerLinkFormatter extends FormatterBase {

  /**
   * HTTP client used for remote favicon API requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * File repository service for storing files.
   *
   * @var \Drupal\file\FileRepository
   */
  protected $fileService;

  /**
   * File system service for directory and file operations.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Logger channel for this formatter.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Messenger service for user-facing error messages.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new formatter instance.
   *
   * Services are injected to keep the class testable and compliant
   * with Drupal best practices.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ClientFactory $http_client_factory,
    FileRepository $file_service,
    FileSystemInterface $file_system,
    LoggerChannelFactoryInterface $logger,
    MessengerInterface $messenger,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $label,
      $view_mode,
      $third_party_settings
    );

    // Initialize injected services.
    $this->httpClient = $http_client_factory->fromOptions([]);
    $this->fileService = $file_service;
    $this->fileSystem = $file_system;
    $this->logger = $logger->get('automatic_site_icon_picker');
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('http_client_factory'),
      $container->get('file.repository'),
      $container->get('file_system'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * Builds render elements for each link field item.
   *
   * Steps:
   *  - Ensure icon cache directory exists.
   *  - Extract domain from URL.
   *  - Download favicon if not cached.
   *  - Build themed render array.
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];

    // Directory used to store cached icons.
    $directory = 'public://social-media-icons/';

    // Prepare directory with proper permissions.
    if (!$this->fileSystem->prepareDirectory(
      $directory,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    )) {
      $this->logger->error(
        'Failed to prepare directory @dir',
        ['@dir' => $directory]
      );
      return $elements;
    }

    // Formatter settings controlling display output.
    $view_mode = $this->getSetting('social_media_link');
    $size_key = $this->getSetting('social_media_size');
    $target = $this->getSetting('social_media_target');

    // Available icon size mappings.
    $sizes = [
      'large' => 64,
      'medium' => 32,
      'small' => 16,
    ];
    $size = $sizes[$size_key] ?? 32;

    // Iterate over all field items.
    foreach ($items as $delta => $item) {

      // Normalize URL: convert internal/entity URLs to absolute form.
      try {
        if (str_starts_with($item->uri, 'internal:') || str_starts_with($item->uri, 'entity:')) {
          $url = Url::fromUri($item->uri, ['absolute' => TRUE])->toString();
        }
        else {
          $url = $item->uri;
        }
      }
      catch (\Exception $e) {
        // Skip invalid or unresolvable URLs.
        $this->logger->warning(
          'Invalid URL @url skipped. Reason: @message',
          ['@url' => $item->uri, '@message' => $e->getMessage()]
        );
        continue;
      }

      // Extract domain name for API request.
      $domain = parse_url($url, PHP_URL_HOST);
      if (!$domain) {
        $this->logger->warning('URL @url has no valid domain.', ['@url' => $url]);
        continue;
      }

      // Build a cache-safe filename.
      $safe_domain = str_replace('.', '-', $domain);
      $filename = $safe_domain . '-' . $size . 'px.png';
      $filepath = $directory . $filename;

      // Download favicon only if not already cached.
      if (!file_exists($filepath)) {
        $data = $this->apiResponse($domain, $size);

        if ($data) {
          try {
            // Save the raw binary data as a file.
            $this->fileSystem->saveData(
              $data,
              $filepath,
              FileExists::Replace,
            );
          }
          catch (\Exception $e) {
            $this->logger->error(
              'Failed to save favicon for @domain. Error: @message',
              ['@domain' => $domain, '@message' => $e->getMessage()]
            );
            continue;
          }
        }
        else {
          // No data returned by API; skip entry.
          $this->logger->warning(
            'Favicon API returned no data for @domain.',
            ['@domain' => $domain]
          );
          continue;
        }
      }

      // Build the render array for the theme template.
      $elements[$delta] = [
        '#theme' => 'automatic_site_icon_picker',
        '#url' => $url,
        '#icon_file' => $filepath,
        '#elem' => [
          'title' => $item->title ?? '',
        ],
        '#view_mode' => $view_mode,
        '#target' => $target,
      ];
    }

    return $elements;
  }

  /**
   * Fetches favicon binary data from the external API.
   *
   * @param string $domain
   *   Domain name such as "example.com".
   * @param int $size
   *   Icon size in pixels.
   *
   * @return string|null
   *   Raw image data on success, NULL on failure.
   */
  protected function apiResponse($domain, $size) {
    try {
      // Remote API request for favicon.
      $response = $this->httpClient->get(
        "https://favicon.vemetric.com/$domain?format=png&size=$size"
      );

      return $response->getBody()->getContents();
    }
    catch (\Exception $e) {
      // Log error but avoid interrupting page render.
      $this->logger->error(
        'Favicon API error for @domain. Message: @message',
        ['@domain' => $domain, '@message' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'social_media_link' => 'icon_only',
      'social_media_size' => 'medium',
      'social_media_target' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   *
   * Adds settings for selected icon display mode and icon size.
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::settingsForm($form, $form_state);

    $form['social_media_link'] = [
      '#title' => $this->t('Display mode'),
      '#type' => 'select',
      '#options' => [
        'icon_only' => $this->t('Icon only'),
        'icon_and_link' => $this->t('Icon and URL'),
        'icon_and_title' => $this->t('Icon and Title'),
      ],
      '#default_value' => $this->getSetting('social_media_link'),
    ];

    $form['social_media_size'] = [
      '#title' => $this->t('Icon size'),
      '#type' => 'select',
      '#options' => [
        'large' => $this->t('64px'),
        'medium' => $this->t('32px'),
        'small' => $this->t('16px'),
      ],
      '#default_value' => $this->getSetting('social_media_size'),
    ];

    $form['social_media_target'] = [
      '#title' => $this->t('Open link in new window'),
      '#type' => 'checkbox',
      '#return_value' => '_blank',
      '#default_value' => $this->getSetting('social_media_target'),
    ];

    return $form;
  }

}
