<?php
namespace Drupal\aws_manager\Form;

use Aws\Exception\AwsException;
use Aws\Organizations\OrganizationsClient;
use Aws\S3\S3Client;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

class AwsAccountViewForm extends FormBase
{

    protected $accountId;

    public function getFormId()
    {
        return 'aws_account_view_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state, $account_id = null)
    {
        $form['#prefix'] = '<div class="aws-account-view-form-wrapper"><div class="inner-wrapper">';
        $form['#suffix'] = '</div></div>';

        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $this->accountId                = $account_id;
        $conn                           = Database::getConnection();

        // Load AWS credentials
        $creds = $conn->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', \Drupal::currentUser()->id())
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $form['error'] = ['#markup' => $this->t('AWS credentials not found.')];
            return $form;
        }

        // Initialize AWS clients
        try {
            $org_client = new OrganizationsClient([
                'version'     => '2016-11-28',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $creds->access_key,
                    'secret' => $creds->secret_key,
                ],
            ]);

            $s3_client = new S3Client([
                'version'     => 'latest',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $creds->access_key,
                    'secret' => $creds->secret_key,
                ],
            ]);

            $result = $org_client->describeAccount([
                'AccountId' => $account_id,
            ]);
            $account = $result->get('Account');
        } catch (AwsException $e) {
            $error_msg  = $e->getAwsErrorMessage() ?: $e->getMessage();
            $error_code = $e->getAwsErrorCode();

            if ($error_code == 'InvalidSignatureException') {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">❌ Invalid AWS credentials. Please <a href="/admin/aws/credentials">check your keys</a> or use the <strong>Test Connection</strong> button to verify them.</div>',
                ];
            } else {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">AWS error: ' . $error_msg . '</div>',
                ];
            }

        } catch (\Exception $e) {

            $form['error'] = ['#markup' => '<div class="messages messages--error">AWS error: ' . $e->getMessage() . '</div>'];
            return $form;
        }

        // View-only AWS account info
        $form['account_info'] = [
            '#type'  => 'details',
            '#title' => $this->t('Account Information'),
            '#open'  => true,
        ];
        $form['account_info']['account_id'] = [
            '#type'   => 'item',
            '#title'  => $this->t('Account ID'),
            '#markup' => $account['Id'],
        ];
        $form['account_info']['account_name'] = [
            '#type'   => 'item',
            '#title'  => $this->t('Account Name'),
            '#markup' => $account['Name'],
        ];
        $form['account_info']['email'] = [
            '#type'   => 'item',
            '#title'  => $this->t('Root Email'),
            '#markup' => $account['Email'],
        ];
        $form['account_info']['status'] = [
            '#type'   => 'item',
            '#title'  => $this->t('Status'),
            '#markup' => $account['Status'],
        ];
        $form['account_info']['joined'] = [
            '#type'   => 'item',
            '#title'  => $this->t('Joined Date'),
            '#markup' => date('Y-m-d H:i:s', strtotime($account['JoinedTimestamp'])),
        ];

        // S3 buckets section
        try {
            $buckets_result = $s3_client->listBuckets();
            $buckets        = $buckets_result->get('Buckets');

            $rows = [];
            foreach ($buckets as $bucket) {
                $rows[] = [
                    $bucket['Name'],
                    date('Y-m-d H:i:s', strtotime($bucket['CreationDate'])),
                ];
            }

            $form['s3_buckets'] = [
                '#type'  => 'details',
                '#title' => $this->t('S3 Buckets (Shared Credential Scope)'),
                '#open'  => true,
            ];
            $form['s3_buckets']['table'] = [
                '#type'   => 'table',
                '#header' => [$this->t('Bucket Name'), $this->t('Created On')],
                '#rows'   => $rows,
                '#empty'  => $this->t('No buckets found or accessible.'),
            ];
        } catch (AwsException $e) {
            $error_msg  = $e->getAwsErrorMessage() ?: $e->getMessage();
            $error_code = $e->getAwsErrorCode();

            if ($error_code == 'InvalidSignatureException') {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">❌ Invalid AWS credentials. Please <a href="/admin/aws/credentials">check your keys</a> or use the <strong>Test Connection</strong> button to verify them.</div>',
                ];
            } else {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">AWS error: ' . $error_msg . '</div>',
                ];
            }

        } catch (\Exception $e) {
            $form['s3_buckets']['error'] = ['#markup' => '<div class="messages messages--warning">Could not fetch buckets: ' . $e->getMessage() . '</div>'];
        }

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        // No submit action needed
    }

}
