<?php
namespace Drupal\aws_manager\Form;

use Aws\S3\S3Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\file\Entity\File;

class S3FileEditForm extends FormBase
{

    protected $bucket;
    protected $key;

    public function getFormId(): string
    {
        return 's3_file_edit_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state, $bucket = null, $key = null): array
    {
        $this->bucket = $bucket;
        $this->key    = $key;

        $form['#prefix'] = '<div id="s3-edit-form-wrapper">';
        $form['#suffix'] = '</div>';

        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $uid                            = $this->currentUser()->id();
        $db                             = \Drupal::database();
        $creds                          = $db->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', $uid)
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $form['error'] = ['#markup' => '<div class="messages messages--error">AWS credentials not found.</div>'];
            return $form;
        }

        $s3 = new S3Client([
            'version'     => 'latest',
            'region'      => 'us-east-1',
            'credentials' => [
                'key'    => $creds->access_key,
                'secret' => $creds->secret_key,
            ],
        ]);

        $download_url = $s3->createPresignedRequest(
            $s3->getCommand('GetObject', ['Bucket' => $bucket, 'Key' => $key]),
            '+5 minutes'
        )->getUri();

        $ext            = strtolower(pathinfo($key, PATHINFO_EXTENSION));
        $preview_markup = in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp'])
        ? '<img src="' . $download_url . '" style="max-width:100%; max-height:200px;" alt="Preview">'
        : '<a href="' . $download_url . '" target="_blank">Preview File</a>';

        $form['preview'] = [
            '#type'   => 'markup',
            '#markup' => Markup::create('<div class="mb-3">' . $preview_markup . '</div>'),
        ];

        $form['replace_file'] = [
            '#type'        => 'file',
            '#title'       => $this->t('Replace with new file'),
            '#description' => $this->t('This will overwrite the existing file.'),
            '#required'    => true,
        ];

        $form['actions']           = ['#type' => 'actions'];
        $form['actions']['submit'] = [
            '#type'        => 'submit',
            '#value'       => $this->t('Upload & Replace'),
            '#button_type' => 'primary',
            '#attributes'  => [
                'class' => ['aws-btn', 'aws-btn-save'],
            ],
            '#ajax'        => [
                'callback' => '::ajaxSubmitHandler',
                'wrapper'  => 's3-edit-form-wrapper',
                'progress' => [
                    'type'    => 'throbber',
                    'message' => $this->t('Uploading...'),
                ],
            ],
        ];

        return $form;
    }

    public function ajaxSubmitHandler(array &$form, FormStateInterface $form_state): AjaxResponse
    {
        $response = new AjaxResponse();

        $file_upload = \Drupal::request()->files->get('files')['replace_file'] ?? null;

        if (! $file_upload || $file_upload->getError()) {
            $response->addCommand(new HtmlCommand('#s3-edit-form-wrapper', '<div class="messages messages--error">File upload failed.</div>'));
            return $response;
        }

        $destination = 'public://s3_temp_uploads';
        \Drupal::service('file_system')->prepareDirectory($destination, \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY | \Drupal\Core\File\FileSystemInterface::MODIFY_PERMISSIONS);

        $filename         = $file_upload->getClientOriginalName();
        $destination_path = $destination . '/' . $filename;
        move_uploaded_file($file_upload->getPathname(), \Drupal::service('file_system')->realpath($destination_path));

        $file = File::create(['uri' => $destination_path]);
        $file->save();

        $file_path = \Drupal::service('file_system')->realpath($file->getFileUri());

        $uid   = $this->currentUser()->id();
        $db    = \Drupal::database();
        $creds = $db->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', $uid)
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $response->addCommand(new HtmlCommand('#s3-edit-form-wrapper', '<div class="messages messages--error">AWS credentials not found.</div>'));
            return $response;
        }

        $s3 = new S3Client([
            'version'     => 'latest',
            'region'      => 'us-east-1',
            'credentials' => [
                'key'    => $creds->access_key,
                'secret' => $creds->secret_key,
            ],
        ]);

        try {
            $s3->putObject([
                'Bucket'     => $this->bucket,
                'Key'        => $this->key,
                'SourceFile' => $file_path,
                'ACL'        => 'private',
            ]);

            $response->addCommand(new HtmlCommand('#s3-edit-form-wrapper', '<div class="messages messages--status">File replaced successfully.</div>'));
            $response->addCommand(new RedirectCommand("/aws/s3/{$this->bucket}/browse"));

        } catch (\Exception $e) {
            $response->addCommand(new HtmlCommand('#s3-edit-form-wrapper', '<div class="messages messages--error">Upload failed: ' . $e->getMessage() . '</div>'));
        }

        return $response;
    }

    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        // Intentionally empty – handled via AJAX.
    }

}
