<?php
namespace Drupal\aws_manager\Form;

use Aws\S3\S3Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\Core\Url;
class S3FileUploadForm extends FormBase
{

    public function getFormId(): string
    {
        return 's3_file_upload_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state, $bucket = null): array
    {
        if (! $bucket) {
            $this->messenger()->addError('Bucket name not provided in URL.');
            return $form;
        }
        $form['#prefix']                = '<div id="s3-upload-wrapper">';
        $form['#suffix']                = '</div>';
        $form['#attached']['library'][] = 'aws_manager/aws_ui';
           $form['live_page'] = [
          '#type'       => 'link',
          '#title'      => $this->t('Live List '),
          '#url'        => Url::fromRoute('aws_manager.aws_live_list'),
          '#attributes' => [
            'class' => ['button', 'aws-btn', 'aws-btn-delete'],
          ],
        ];
        $form['bucket']                 = [
            '#type'   => 'item',
            '#title'  => $this->t('Target S3 Bucket'),
            '#markup' => $bucket,
        ];

        $form['bucket_hidden'] = [
            '#type'  => 'hidden',
            '#value' => $bucket,
        ];

        $form['file'] = [
            '#type'            => 'managed_file',
            '#title'           => $this->t('Choose files'),
            '#description'     => $this->t('Select multiple files to upload to S3.'),
            '#upload_location' => 'public://',
            '#multiple'        => true,
            '#required'        => true,
        ];

        $form['s3_save_aws']['save'] = [
            '#type'       => 'submit',
            '#value'      => $this->t('Save to S3 storage'),
            '#name'       => 's3_upload_submit',
            '#ajax'       => [
                'callback' => '::ajaxSubmitCallback',
                'wrapper'  => 's3-upload-wrapper',
                'progress' => [
                    'type'    => 'throbber',
                    'message' => $this->t('Uploading...'),
                ],
            ],
            '#attributes' => [
                'class'       => ['aws-btn', 'aws-btn-save', 's3-upload-btn'],
                'data-action' => 'upload',
            ],
        ];

        return $form;
    }
    public function ajaxSubmitCallback(array &$form, FormStateInterface $form_state): AjaxResponse
    {
        $bucket  = $form_state->getValue('bucket_hidden');
        $user_id = $this->currentUser()->id();
        $conn    = \Drupal::database();

        $creds = $conn->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', $user_id)
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $this->messenger()->addError('AWS credentials not found.');
            return new AjaxResponse();
        }

        $s3 = new S3Client([
            'version'     => 'latest',
            'region'      => 'us-east-1',
            'credentials' => [
                'key'    => $creds->access_key,
                'secret' => $creds->secret_key,
            ],
        ]);

        try {
            if (! $s3->doesBucketExist($bucket)) {
                $s3->createBucket(['Bucket' => $bucket]);
                $s3->waitUntil('BucketExists', ['Bucket' => $bucket]);
            }
        } catch (\Exception $e) {
            $this->messenger()->addError('Bucket creation failed: ' . $e->getMessage());
            return new AjaxResponse();
        }

        $file_ids = $form_state->getValue('file');
        if (empty($file_ids) || ! is_array($file_ids)) {
            $this->messenger()->addError('No files selected.');
            return new AjaxResponse();
        }

        $fs = \Drupal::service('file_system');

        foreach ($file_ids as $fid) {
            $file = File::load($fid);
            if (! $file) {
                continue;
            }

            $file->setPermanent();
            $file->save();

            $uri       = $file->getFileUri();
            $real_path = $fs->realpath($uri);
            $filename  = $file->getFilename();
            $file_size = filesize($real_path);

            try {
                $s3_result = $s3->putObject([
                    'Bucket'     => $bucket,
                    'Key'        => $filename,
                    'SourceFile' => $real_path,
                    'ACL'        => 'private',
                ]);
             
                $conn->insert('aws_files')
                    ->fields([
                        'file_name'        => $filename,
                        'uid'              => \Drupal::currentUser()->id(),
                        'fid'              => $fid,
                        'file_type'        => pathinfo($filename, PATHINFO_EXTENSION),
                        'alt_tag'          => pathinfo($filename, PATHINFO_FILENAME),
                        'main_image'       => 0,
                        'uploaded_by'      => $user_id,
                        'date_uploaded'    => date('Y-m-d H:i:s'),
                        'file_size'        => $file_size,
                        'aws_folder_id'    => null,
                        'module_entity_id' => null,
                    ])
                    ->execute();

                $this->messenger()->addStatus("✅ Uploaded to S3: " . $s3_result->get('ObjectURL'));

            } catch (\Exception $e) {
                $this->messenger()->addError("❌ Upload failed for {$filename}: " . $e->getMessage());
            }
        }

        // Redirect to current page after successful upload
        $response     = new AjaxResponse();
        $current_path = \Drupal::service('path.current')->getPath(); // e.g. /some/path
        $response->addCommand(new RedirectCommand($current_path));
        return $response;
    }

    public function submitForm(array &$form, FormStateInterface $form_state): void
    {}

}
