# AWS Manager for Drupal

The **AWS Manager** module provides a complete backend interface for managing AWS services within a Drupal 10 environment. It includes:

- ✅ Secure credential storage per Drupal user  
- ✅ AWS Organizations account management  
- ✅ Full-featured S3 bucket and file control (create/edit/delete/upload/browse)  
- ✅ AJAX-based file upload UI  
- ✅ Modal-driven editing and deletion for both buckets and files  
- ✅ Tabbed interface for file type filtering: Documents, Images, and Others  

---

## 🚀 Requirements

This module depends on the **official AWS PHP SDK**.

### ✅ Install AWS SDK via Composer

You **must** install the AWS SDK using Composer before enabling the module:

```bash
composer require aws/aws-sdk-php
```

The SDK provides integration with:
- `OrganizationsClient` (to fetch accounts)
- `S3Client` (for buckets and files)

---

## 🔧 Installation and Configuration

### Step 1: Enable the Module

```bash
drush en aws_manager
```

Or enable via the Drupal UI.

---

### Step 2: Configure AWS Credentials

Each user must enter their AWS credentials manually.

- Navigate to: `Configuration → AWS Manager → AWS Credentials`
- Or visit: `/admin/aws/credentials`

Fields include:

- **Access Key**
- **Secret Key**

These credentials are securely stored and associated with the current Drupal user account only.

---

### Step 3: Manage AWS Organization Clients

Navigate to:

- `Configuration → AWS Manager → Manage AWS Clients`
- Or `/admin/aws/client`

You can:

- Register AWS Organization accounts
- View existing accounts
- View, edit, or delete them via modal dialogs

---

### Step 4: Use the Live List Dashboard

Visit `/admin/aws/live-list` or click the **Live AWS Accounts & Buckets** link in the menu.

You will see:

- A vertical tabbed layout
- AWS Organization accounts
- A list of your S3 buckets fetched via AWS API
- For each item: View, Delete, Browse Files, and Upload buttons

---

## 📁 S3 Bucket Management

### Bucket Operations

Each bucket entry shows:

- Bucket Name
- Creation Date
- AWS-accessible URL
- Button: **View Files** (modal or page)
- Button: **Delete Bucket** (modal confirmation)

Buckets are created automatically if they don't exist during upload.

---

### File Uploading

To upload files:

- Go to `/admin/aws/upload-file/{bucket}`
- Select one or more files using the `managed_file` field
- Files are uploaded directly to the S3 bucket
- File metadata is stored in the `aws_files` table
- File types are automatically categorized (see tabs below)

---

### File Browsing

When viewing a bucket:

- Files are shown under 3 tabbed views:
  - **Documents** (PDF, DOCX, XLSX, etc.)
  - **Images** (JPG, PNG, GIF, SVG, etc.)
  - **Others** (ZIP, EXE, JSON, etc.)

Each file has options to:

- View
- Edit
- Delete

All of these actions are AJAX modal-based.

---

## 🔐 Permissions

The module uses one permission:

```
access aws manager
```

Only users with this permission can access the credentials form, bucket dashboards, upload tools, and delete modals.

---

## 🌐 Menu Integration

All AWS Manager admin links appear under:

**Configuration → AWS Manager**

Menu includes:

- AWS Credentials
- Manage AWS Clients
- Live AWS Accounts & Buckets

> Note: Routes like file upload or bucket browsing require parameters and are **not shown** as menu items — they are accessed through table actions and modals.

---

## 🛠 Developer Notes

### 📊 Tables Created

- `aws_credentials` — stores user-specific access/secret keys
- `aws_files` — stores file metadata for uploaded S3 files

### 📁 Entities Used

- `file` — Drupal's core file entity is used for handling local uploads before pushing to S3

---

## 💻 Developer Hooks

### `hook_aws_manager_upload_file_alter()`

Allows you to customize the file metadata before saving to the database.

```php
/**
 * Alter file metadata before it's saved.
 */
function mymodule_aws_manager_upload_file_alter(array &$record, \Drupal\file\FileInterface $file) {
  // Append suffix to alt tag
  $record['alt_tag'] .= ' (Imported)';
}
```

---

## 🧼 Best Practices

- Protect all `/admin/aws/*` routes using IP or Shield module
- Assign `access aws manager` only to admin/editorial roles
- Limit the AWS keys used to least privilege permissions:
  - `s3:PutObject`, `s3:DeleteObject`, `s3:ListBucket`
  - `organizations:ListAccounts` for client fetching

---

## 👨‍💻 Maintainer

Developed and maintained by your internal Drupal team.

Pull requests, issues, and suggestions are always welcome.

Need help? Reach out at: `krishanand46@gmail.com`
---

