<?php
namespace Drupal\aws_manager\Form;

use Aws\Exception\AwsException;
use Aws\Organizations\OrganizationsClient;
use Aws\Sts\StsClient;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

class AwsAccountDeleteForm extends ConfirmFormBase
{

    protected $accountId;

    public function getFormId()
    {
        return 'aws_account_delete_form';
    }

    public function getQuestion()
    {
        return $this->t('Are you sure you want to delete the AWS account %id?', ['%id' => $this->accountId]);
    }

    public function getCancelUrl()
    {
        return new Url('aws_manager.aws_live_list');
    }

    public function getConfirmText()
    {
        return $this->t('Delete');
    }

    public function buildForm(array $form, FormStateInterface $form_state, $account_id = null)
    {
        $this->accountId = $account_id;
        return parent::buildForm($form, $form_state);
    }

    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $database = Database::getConnection();

        $creds = $database->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', \Drupal::currentUser()->id())
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $this->messenger()->addError('AWS credentials not found.');
            return;
        }

        $aws_config = [
            'version'     => '2016-11-28',
            'region'      => 'us-east-1',
            'credentials' => [
                'key'    => $creds->access_key,
                'secret' => $creds->secret_key,
            ],
        ];

        try {
            $org_client = new OrganizationsClient($aws_config);

            $sts_client = new StsClient([
                'version'     => '2011-06-15',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $creds->access_key,
                    'secret' => $creds->secret_key,
                ],
            ]);

            $caller_identity    = $sts_client->getCallerIdentity();
            $current_account_id = $caller_identity['Account'];

            if ($current_account_id === $this->accountId) {
                $org_client->leaveOrganization();
                $this->messenger()->addStatus("✅ This member account {$this->accountId} has successfully left the organization.");
            } else {
                $org_client->removeAccountFromOrganization([
                    'AccountId' => $this->accountId,
                ]);
                $this->messenger()->addStatus("✅ AWS member account {$this->accountId} removed from organization by management account.");
            }

            // ------------------------------
            // 🧹 Clean up from local tables
            // ------------------------------
            $client_row = $database->select('aws_client_accounts', 'c')
                ->fields('c', ['company_id'])
                ->condition('aws_client_acccount_id', $this->accountId)
                ->range(0, 1)
                ->execute()
                ->fetchAssoc();
            $company_id = $client_row['company_id'] ?? null;

            $database->delete('aws_client_accounts')
                ->condition('aws_client_acccount_id', $this->accountId)
                ->execute();

            if ($company_id) {
                $folder_ids = $database->select('aws_folders', 'f')
                    ->fields('f', ['aws_folder_id'])
                    ->condition('company_id', $company_id)
                    ->execute()
                    ->fetchCol();

                if (! empty($folder_ids)) {
                    $database->delete('aws_files')
                        ->condition('aws_folder_id', $folder_ids, 'IN')
                        ->execute();
                }

                $database->delete('aws_folders')
                    ->condition('company_id', $company_id)
                    ->execute();
            }

            $database->delete('aws_clients')
                ->condition('aws_account_id', $this->accountId)
                ->execute();

            $this->messenger()->addStatus("🧹 Local AWS account and associated files/folders removed.");

        } catch (AwsException $e) {
            $code    = $e->getAwsErrorCode();
            $message = $e->getAwsErrorMessage();

            if ($code === 'ConstraintViolationException') {
                $this->messenger()->addError("❌ Account cannot leave yet — missing required info:
- Add credit card
- Choose a support plan
- Complete contact info
- Must be > 7 days old");
            } elseif ($code === 'MasterCannotLeaveOrganizationException') {
                $this->messenger()->addError("❌ This is a management account. You must first delete the organization to detach it.");
            } elseif ($code === 'AccountNotFoundException') {
                $this->messenger()->addError("❌ Member account ID {$this->accountId} not found in the current organization.");
            } elseif ($code == 'InvalidSignatureException') {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">❌ Invalid AWS credentials. Please <a href="/admin/aws/credentials">check your keys</a> or use the <strong>Test Connection</strong> button to verify them.</div>',
                ];
            } else {
                $this->messenger()->addError("AWS error [$code]: $message");
            }

        } catch (\Exception $e) {
            $this->messenger()->addError('Error: ' . $e->getMessage());
        }

        $form_state->setRedirectUrl($this->getCancelUrl());
    }

}
