<?php
namespace Drupal\aws_manager\Form;

use Aws\Exception\AwsException;
use Aws\S3\S3Client;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
class AwsCredentialsForm extends FormBase
{

    public function getFormId()
    {
        return 'aws_credentials_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $uid        = \Drupal::currentUser()->id();
        $connection = Database::getConnection();
        $record     = $connection->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', $uid)
            ->range(0, 1)
            ->execute()
            ->fetchAssoc();

        $form['#prefix'] = '<div id="aws-credentials-wrapper">';
        $form['#suffix'] = '</div>';

        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $form['live_page'] = [
          '#type'       => 'link',
          '#title'      => $this->t('Live List '),
          '#url'        => Url::fromRoute('aws_manager.aws_live_list'),
          '#attributes' => [
            'class' => ['button', 'aws-btn', 'aws-btn-delete'],
          ],
        ];
        $form['access_key']             = [
            '#type'          => 'textfield',
            '#title'         => 'Access Key ID',
            '#default_value' => $record['access_key'] ?? '',
            '#required'      => true,
        ];

        $form['secret_key'] = [
            '#type'          => 'textfield',
            '#title'         => 'Secret Access Key',
            '#default_value' => $record['secret_key'] ?? '',
            '#required'      => true,
        ];

        $form['actions'] = ['#type' => 'actions'];

        $form['actions']['test_connection'] = [
            '#type'                    => 'submit',
            '#value'                   => $this->t('Test Connection'),
            '#name'                    => 'test_connection',
            '#attributes'              => [
                'class' => ['aws-btn', 'aws-btn-delete'],
            ],
            '#ajax'                    => [
                'callback' => '::testConnectionAjax',
                'wrapper'  => 'aws-credentials-wrapper',
            ],
            '#limit_validation_errors' => [['access_key'], ['secret_key']],
            '#submit'                  => ['::testConnectionSubmit'],
        ];

        $form['actions']['submit'] = [
            '#type'       => 'submit',
            '#value'      => $this->t('Save'),
            '#attributes' => [
                'class' => ['aws-btn', 'aws-btn-save'],
            ],
        ];

        return $form;
    }

    /**
     * Ajax callback for Test Connection.
     */
    public function testConnectionAjax(array &$form, FormStateInterface $form_state)
    {
        return $form;
    }

    /**
     * Handles the "Test Connection" AJAX submission.
     */
    public function testConnectionSubmit(array &$form, FormStateInterface $form_state)
    {
        $access_key = $form_state->getValue('access_key');
        $secret_key = $form_state->getValue('secret_key');

        try {
            $s3 = new S3Client([
                'version'     => 'latest',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $access_key,
                    'secret' => $secret_key,
                ],
            ]);

            // Attempt to list buckets as a basic permission check
            $s3->listBuckets();

            $this->messenger()->addStatus('✅ AWS connection successful!');
        } catch (AwsException $e) {
            $this->messenger()->addError('❌ AWS connection failed: ' . $e->getAwsErrorMessage());
        } catch (\Exception $e) {
            $this->messenger()->addError('❌ Error: ' . $e->getMessage());
        }
    }

    /**
     * Final save action.
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $access_key = $form_state->getValue('access_key');
        $secret_key = $form_state->getValue('secret_key');
        $uid        = \Drupal::currentUser()->id();

        $connection = Database::getConnection();

        // Save to aws_credentials table
        $connection->merge('aws_credentials')
            ->key(['uid' => $uid])
            ->fields([
                'access_key' => $access_key,
                'secret_key' => $secret_key,
                'uid'        => $uid,
            ])
            ->execute();

        // Also update or insert into aws_organisation
        $exists = $connection->select('aws_organisation', 'org')
            ->fields('org', ['aws_org_id'])
            ->condition('username', 'user_' . $uid)
            ->range(0, 1)
            ->execute()
            ->fetchField();

        if ($exists) {
            $connection->update('aws_organisation')
                ->fields([
                    'api_key'    => $access_key,
                    'api_secret' => $secret_key,
                ])
                ->condition('aws_org_id', $exists)
                ->execute();
        } else {
            $connection->insert('aws_organisation')
                ->fields([
                    'username'   => 'user_' . $uid,
                    'password'   => 'secret',
                    'uid'        => $uid,
                    'api_key'    => $access_key,
                    'api_secret' => $secret_key,
                ])
                ->execute();
        }

        $this->messenger()->addStatus('AWS credentials saved and organization updated.');
        // $form_state->setRedirect('aws_manager.aws_live_list');
    }

}
