<?php
namespace Drupal\aws_manager\Form;

use Aws\Exception\AwsException;
use Aws\Organizations\OrganizationsClient;
use Aws\S3\S3Client;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

class AwsLiveClientListForm extends FormBase
{

    public function getFormId()
    {
        return 'aws_live_client_list_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $form['#prefix']                = '<div class="aws-live-client-form-wrapper"><div class="inner-content">';
        $form['#suffix']                = '</div></div>';
        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $form['#cache']['max-age']      = 0;

        $database = Database::getConnection();
        $creds    = $database->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', \Drupal::currentUser()->id())
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $form['error'] = [
                '#markup' => '<div class="messages messages--error">AWS credentials not found. <a href="/admin/aws/credentials">Set credentials</a>.</div>',
            ];
            return $form;
        }

        $access_key = $creds->access_key;
        $secret_key = $creds->secret_key;

        $aws_config = [
            'version'     => '2016-11-28',
            'region'      => 'us-east-1',
            'credentials' => [
                'key'    => $access_key,
                'secret' => $secret_key,
            ],
        ];

        try {

            // Tabs wrapper
            $form['tabs'] = [
                '#type'  => 'vertical_tabs',
                '#title' => $this->t('AWS Information'),
            ];

            /** ---------- AWS ACCOUNTS TAB ---------- **/
            $org_client      = new OrganizationsClient($aws_config);
            $accounts_result = $org_client->listAccounts();
            $accounts        = $accounts_result->get('Accounts');

            $form['accounts_section'] = [
                '#type'  => 'details',
                '#title' => $this->t('AWS Organization Accounts'),
                '#group' => 'tabs',
            ];
            $form['accounts_section']['account_modal_container'] = [
                '#type'       => 'container',
                '#attributes' => [
                    'class' => ['aws-account-modal-wrapper'],
                    'style' => 'margin-bottom: 20px;',
                ],
            ];

            $form['accounts_section']['account_modal_container']['header'] = [
                '#markup' => '<h4 class="modal-header-label">Add AWS Account</h4>',
            ];

            $form['accounts_section']['account_modal_container']['modal_link'] = [
                '#type'       => 'link',
                '#title'      => new FormattableMarkup('<i class="fa fa-plus-circle"></i> Add Account AWS', []),
                '#url'        => Url::fromRoute('aws_manager.client_manage_form'),
                '#attributes' => [
                    'class'               => ['use-ajax', 'button', 'btn', 'btn-success', 'custom-off-canvas-modal'],
                    'data-dialog-type'    => 'modal',
                    'data-dialog-options' => json_encode([
                        'width'       => 900,
                        'height'      => 600,
                        'dialogClass' => 'custom-off-canvas-dialog aws-account-view-modal',
                    ]),
                ],
            ];
            $form['accounts_section']['accounts_table'] = [
                '#type'       => 'table',
                '#header'     => [

                    'S.No',
                    'Account Id',
                    'Email',
                    'Name',
                    'Status',
                    'Joined',
                    'View',
                    'Delete',
                ],
                '#empty'      => $this->t('No AWS accounts found.'),
                '#attributes' => ['id' => 'aws-accounts-table'],
            ];

            $sno = 1;
            foreach ($accounts as $acc) {
                $view_url = Url::fromRoute('aws_manager.account_view', ['account_id' => $acc['Id']], [
                    'attributes' => [
                        'class'               => ['use-ajax'],
                        'data-dialog-type'    => 'modal',
                        'data-dialog-options' => Json::encode(['width' => 700]),
                    ],
                ]);

                $view_link['#attached']['library'][] = 'core/drupal.dialog.ajax';
                $view_link                           = Link::fromTextAndUrl('View', $view_url)->toRenderable();
                $view_link['#attributes']['class'][] = 'aws-btn';
                $view_link['#attributes']['class'][] = 'aws-btn-view';

                $delete_url  = Url::fromRoute('aws_manager.account_delete', ['account_id' => $acc['Id']]);
                $delete_link = Link::fromTextAndUrl('Delete', $delete_url)->toRenderable();

                $delete_link['#attributes']['class'][]        = 'aws-btn';
                $delete_link['#attributes']['class'][]        = 'aws-btn-delete';
                $form['accounts_section']['accounts_table'][] = [

                    'Sno'    => ['#markup' => $sno++],
                    'Id'     => ['#markup' => $acc['Id']],
                    'Email'  => ['#markup' => $acc['Email']],
                    'Name'   => ['#markup' => $acc['Name']],
                    'Status' => ['#markup' => $acc['Status']],
                    'Joined' => ['#markup' => date('Y-m-d', strtotime($acc['JoinedTimestamp']))],
                    'View'   => $view_link,
                    'Delete' => $delete_link,
                ];
            }

            /** ---------- S3 BUCKETS TAB ---------- **/
            $s3_client = new S3Client([
                'version'     => 'latest',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $access_key,
                    'secret' => $secret_key,
                ],
            ]);

            $bucket_result = $s3_client->listBuckets();
            $buckets       = $bucket_result->get('Buckets');

            $form['buckets_section'] = [
                '#type'  => 'details',
                '#title' => $this->t('S3 Buckets'),
                '#group' => 'tabs',
            ];

            $form['buckets_section']['buckets_table'] = [
                '#type'   => 'table',
                '#header' => ['Bucket Name', 'Creation Date', 'Bucket URL', 'View Files', 'Delete'],
                '#empty'  => $this->t('No S3 buckets found.'),
            ];

            foreach ($buckets as $bucket) {
                $bucket_name = $bucket['Name'];
                $url         = "https://$bucket_name.s3.amazonaws.com";
                $delete_url  = Url::fromRoute('aws_manager.s3_bucket_delete', ['bucket_name' => $bucket_name], [
                    'attributes' => [
                        'class'               => ['use-ajax'],
                        'data-dialog-type'    => 'modal',
                        'data-dialog-options' => Json::encode(['width' => 500]),
                    ],
                ]);

                $delete_files_link                           = Link::fromTextAndUrl('Delete', $delete_url)->toRenderable();
                $delete_files_link['#attached']['library'][] = 'core/drupal.dialog.ajax';

                $view_files_url  = Url::fromRoute('aws_manager.s3_bucket_files', ['bucket_name' => $bucket_name]);
                $view_files_link = Link::fromTextAndUrl('View Files', $view_files_url)->toRenderable();

                $view_files_link['#attributes']['class'][] = 'aws-btn';
                $view_files_link['#attributes']['class'][] = 'aws-btn-view';

                $delete_files_link['#attributes']['class'][] = 'aws-btn';
                $delete_files_link['#attributes']['class'][] = 'aws-btn-delete';
                $form['buckets_section']['buckets_table'][]  = [
                    'Name'    => ['#markup' => $bucket_name],
                    'Created' => ['#markup' => date('Y-m-d H:i:s', strtotime($bucket['CreationDate']))],
                    'URL'     => ['#markup' => Link::fromTextAndUrl($url, Url::fromUri($url))->toString()],
                    'View'    => $view_files_link,
                    'Delete'  => $delete_files_link,
                ];

            }

        } catch (AwsException $e) {
            $error_msg  = $e->getAwsErrorMessage() ?: $e->getMessage();
            $error_code = $e->getAwsErrorCode();

            if ($error_code == 'InvalidSignatureException') {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">❌ Invalid AWS credentials. Please <a href="/admin/aws/credentials">check your keys</a> or use the <strong>Test Connection</strong> button to verify them.</div>',
                ];
            } else {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">AWS error: ' . $error_msg . '</div>',
                ];
            }

        } catch (\Exception $e) {
            $form['error'] = [
                '#markup' => '<div class="messages messages--error">Unexpected error: ' . $e->getMessage() . '</div>',
            ];
        }
        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        // No submit
    }
}
