<?php
namespace Drupal\aws_manager\Form;

use Aws\Exception\AwsException;
use Aws\Organizations\OrganizationsClient;
use Aws\S3\S3Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
class ClientManageForm extends FormBase
{

    public function getFormId()
    {
        return 'aws_client_manage_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $form['#prefix']                = '<div id="aws-client-form-wrapper">';
        $form['#suffix']                = '</div>';
        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $current_user                   = \Drupal::currentUser();
        $database                       = Database::getConnection();
        $getApiCreds                    = $database->select('aws_credentials', 'was')
            ->fields('was')
            ->condition('uid', $current_user->id())
            ->execute()
            ->fetchObject();

        if (empty($getApiCreds)) {
            $form['aws_cred_error'] = [
                '#type'   => 'markup',
                '#markup' => Markup::create('<div class="messages messages--error"><strong>Missing Credentials:</strong> You haven\'t set your AWS API credentials yet. <a href="/admin/aws/credentials">Click here to set them</a>.</div>'),
            ];
            return $form;
        }
        $form['message'] = [
            '#type'   => 'markup',
            '#markup' => '<div id="aws-client-message"></div>',
        ];

        $form_state->set('aws_access_key', $getApiCreds->access_key);
        $form_state->set('aws_secret_key', $getApiCreds->secret_key);
   $form['live_page'] = [
          '#type'       => 'link',
          '#title'      => $this->t('Live List '),
          '#url'        => Url::fromRoute('aws_manager.aws_live_list'),
          '#attributes' => [
            'class' => ['button', 'aws-btn', 'aws-btn-delete'],
          ],
        ];
        $form['create_account'] = [
            '#type'          => 'checkbox',
            '#title'         => $this->t('Create new AWS account for client'),
            '#default_value' => true,
            '#description'   => $this->t('Check this if you want to create a new AWS account along with the S3 bucket. <strong>If the client already has an AWS account, uncheck this to only create a new S3 bucket</strong>.'),
            '#ajax'          => [
                'callback' => '::toggleClientFields',
                'wrapper'  => 'client-fields-wrapper',
                'effect'   => 'fade',
            ],
        ];

        $form['client_fields'] = [
            '#type'       => 'container',
            '#attributes' => ['id' => 'client-fields-wrapper'],
        ];

        $create_account = $form_state->getValue('create_account', true);

        $form['client_fields']['client_name'] = [
            '#type'     => 'textfield',
            '#title'    => 'Client Name',
            '#required' => $create_account,
            '#disabled' => ! $create_account,
            '#states'   => [
                'visible' => [
                    ':input[name="create_account"]' => ['checked' => true],
                ],
            ],
        ];

        $form['client_fields']['client_email'] = [
            '#type'        => 'email',
            '#title'       => 'Client Email',
            '#required'    => $create_account,
            '#description' => 'This email will receive AWS account creation and verification steps.',
            '#disabled'    => ! $create_account,
            '#states'      => [
                'visible' => [
                    ':input[name="create_account"]' => ['checked' => true],
                ],
            ],
        ];

        $form['bucket_name'] = [
            '#type'             => 'textfield',
            '#title'            => 'S3 Bucket Name',
            '#required'         => true,
            '#element_validate' => [[$this, 'validateBucketName']],
        ];

        $form['submit'] = [
            '#type'       => 'submit',
            '#value'      => 'Create AWS Resource(s)',
            '#attributes' => [
                'class' => ['aws-btn', 'aws-btn-save'],
            ],
            '#ajax'       => [
                'callback' => '::submitAjaxCallback',
                'wrapper'  => 'aws-client-message',
                'progress' => ['type' => 'throbber', 'message' => 'Processing...'],
            ],
        ];

        return $form;
    }

    public function toggleClientFields(array &$form, FormStateInterface $form_state)
    {
        return $form['client_fields'];
    }

    public function submitAjaxCallback(array &$form, FormStateInterface $form_state)
    {
        $response = new AjaxResponse();

        $create_account  = $form_state->getValue('create_account');
        $bucket_name     = strtolower($form_state->getValue('bucket_name') . '-' . time());
        $aws_access_key  = $form_state->get('aws_access_key');
        $aws_secret_key  = $form_state->get('aws_secret_key');
        $current_user_id = \Drupal::currentUser()->id();

        try {
            $aws_config = [
                'version'     => '2016-11-28',
                'region'      => 'us-east-1',
                'credentials' => ['key' => $aws_access_key, 'secret' => $aws_secret_key],
            ];

            $account_id = null;

            if ($create_account) {
                $org_client = new OrganizationsClient($aws_config);
                $create     = $org_client->createAccount([
                    'AccountName'            => $form_state->getValue('client_name'),
                    'Email'                  => $form_state->getValue('client_email'),
                    'IamUserAccessToBilling' => 'ALLOW',
                    'RoleName'               => 'OrgAccountAccessRole',
                ]);
                $request_id = $create['CreateAccountStatus']['Id'];
                sleep(10);
                $status     = $org_client->describeCreateAccountStatus(['CreateAccountRequestId' => $request_id]);
                $account_id = $status['CreateAccountStatus']['AccountId'] ?? null;
            }
            $uid = \Drupal::currentUser()->id();
                    

            $s3 = new S3Client([
                'version'     => 'latest',
                'region'      => $aws_config['region'],
                'credentials' => ['key' => $aws_access_key, 'secret' => $aws_secret_key],
            ]);

            $bucket_params = ['Bucket' => $bucket_name];
            if ($aws_config['region'] !== 'us-east-1') {
                $bucket_params['CreateBucketConfiguration'] = ['LocationConstraint' => $aws_config['region']];
            }

            $s3->createBucket($bucket_params);
            $s3->waitUntil('BucketExists', ['Bucket' => $bucket_name]);

            $conn   = Database::getConnection();
            $org_id = $conn->insert('aws_organisation')->fields([
                'username'   => $form_state->getValue('client_name') ?: 'NoAccount',
                'password'   => 'secret',
                'api_key'    => $aws_access_key,
                'api_secret' => $aws_secret_key,
                'uid'        => $uid,
            ])->execute();

            $conn->insert('aws_client_accounts')->fields([
                'aws_organisation' => $org_id,
                'type'             => 'CRM',
                'password'         => 'generated',
                'api_key'          => $aws_access_key,
                'api_secret'       => $aws_secret_key,
                'uid'        => $uid,
            ])->execute();

            $conn->insert('aws_folders')->fields([
                'module_name'  => 'crm',
                'added_by_uid' => $current_user_id,
                'path'         => $bucket_name,
                'uid'        => $uid,
            ])->execute();

            // $conn->insert('aws_files')->fields([
            //     'file_name'        => '.placeholder',
            //     'file_type'        => 'system',
            //     'alt_tag'          => 'root',
            //     'main_image'       => 0,
            //     'uploaded_by'      => $current_user_id,
            //     'date_uploaded'    => date('Y-m-d H:i:s'),
            //     'file_size'        => 0,
            //     'aws_folder_id'    => $folder_id,
            //     'module_entity_id' => 0,
            // ])->execute();
            // dd($create_account);
            $msg = $create_account
            ? " AWS account and S3 bucket created successfully!"
            : " S3 bucket created without new AWS account.";
            $stat = ($create_account == 0) ? 'warning' : 'status';
            $response->addCommand(new MessageCommand($msg, $stat));

        } catch (AwsException $e) {
            $error = $e->getAwsErrorMessage();
            $response->addCommand(new MessageCommand($error, 'error'));

            $response->addCommand(new HtmlCommand('#aws-client-message', "<div class='messages messages--error'>AWS Error: {$error}</div>"));
        } catch (\Exception $e) {
            $response->addCommand(new HtmlCommand('#aws-client-message', "<div class='messages messages--error'>General Error: {$e->getMessage()}</div>"));
        }

        return $response;
    }

    public function validateBucketName(&$element, FormStateInterface $form_state, &$complete_form)
    {
        $bucket = $form_state->getValue('bucket_name');
        if (! preg_match('/^[a-z0-9][a-z0-9.-]{1,61}[a-z0-9]$/', $bucket)) {
            $form_state->setError($element, $this->t('Invalid bucket name. Use lowercase letters, numbers, and hyphens only.'));
        }
    }
    public function submitForm(array &$form, FormStateInterface $form_state)
    {

    }
}
