<?php
namespace Drupal\aws_manager\Form;

use Aws\S3\S3Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

class DeleteS3BucketForm extends FormBase
{

    public function getFormId()
    {
        return 'aws_delete_s3_bucket_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state, $bucket_name = null)
    {
        $form['bucket_name'] = [
            '#type'  => 'value',
            '#value' => $bucket_name,
        ];
        $form['#attached']['library'][] = 'aws_manager/aws_ui';
        $form['message']                = [
            '#markup' => "<p>Are you sure you want to delete the bucket: <strong>{$bucket_name}</strong>?</p>",
        ];

        $form['actions']['submit'] = [
            '#type'        => 'submit',
            '#value'       => $this->t('Yes, Delete'),
            '#button_type' => 'danger',
            '#ajax'        => ['callback' => '::submitAjax', 'wrapper' => 'aws-delete-modal-wrapper'],
        ];

        $form['actions']['cancel'] = [
            '#type'       => 'button',
            '#value'      => $this->t('Cancel'),
            '#attributes' => ['class' => ['dialog-cancel']],
        ];

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state)
    {}

    public function submitAjax(array &$form, FormStateInterface $form_state)
    {
        $bucket_name = $form_state->getValue('bucket_name');
        $conn        = \Drupal::database();

        try {
            $creds = $conn->select('aws_credentials', 'c')
                ->fields('c')
                ->condition('uid', \Drupal::currentUser()->id())
                ->execute()->fetchObject();

            $s3 = new S3Client([
                'version'     => 'latest',
                'region'      => 'us-east-1',
                'credentials' => [
                    'key'    => $creds->access_key,
                    'secret' => $creds->secret_key,
                ],
            ]);

            $objects = $s3->listObjectsV2(['Bucket' => $bucket_name]);

            if (! empty($objects['Contents'])) {
                foreach ($objects['Contents'] as $object) {
                    $s3->deleteObject(['Bucket' => $bucket_name, 'Key' => $object['Key']]);
                }
            }

            $s3->deleteBucket(['Bucket' => $bucket_name]);

            $response = new AjaxResponse();
            $response->addCommand(new CloseModalDialogCommand());
            $url = Url::fromRoute('aws_manager.aws_live_list')->toString();
            $response->addCommand(new RedirectCommand($url));
            $response->addCommand(new MessageCommand("Bucket '{$bucket_name}' deleted successfully.", null, ['type' => 'status']));
            return $response;

        } catch (\Exception $e) {
            $response = new AjaxResponse();
            $response->addCommand(new MessageCommand("Error deleting bucket: " . $e->getMessage(), null, ['type' => 'error']));
            return $response;
        }
    }
}
