<?php
namespace Drupal\aws_manager\Form;

use Aws\S3\Exception\S3Exception;
use Aws\S3\S3Client;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;

class S3BucketFileBrowserForm extends FormBase
{

    public function getFormId(): string
    {
        return 's3_bucket_file_browser_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state, $bucket_name = null): array
    {
        $form['#prefix']                = '<div id="s3-bucket-file-upload-wrapper">';
        $form['#suffix']                = '</div>';
        $form['#attached']['library'][] = 'aws_manager/aws_ui';

        $form['#title']            = $this->t('Files in S3 Bucket: @bucket', ['@bucket' => $bucket_name]);
        $form['#cache']['max-age'] = 0;

        $uid   = $this->currentUser()->id();
        $db    = \Drupal::database();
        $creds = $db->select('aws_credentials', 'c')
            ->fields('c')
            ->condition('uid', $uid)
            ->execute()
            ->fetchObject();

        if (! $creds) {
            $form['error'] = [
                '#markup' => '<div class="messages messages--error">AWS credentials not found. <a href="/admin/aws/credentials">Set credentials</a>.</div>',
            ];
            return $form;
        }

        $default_region = 'us-east-1';
        $s3             = new S3Client([
            'version'     => 'latest',
            'region'      => $default_region,
            'credentials' => [
                'key'    => $creds->access_key,
                'secret' => $creds->secret_key,
            ],
        ]);

        try {
            $result = $s3->listObjectsV2(['Bucket' => $bucket_name]);
        } catch (S3Exception $e) {
            if ($e->getAwsErrorCode() === 'AuthorizationHeaderMalformed' && strpos($e->getAwsErrorMessage(), 'expecting') !== false) {
                preg_match('/expecting \'([^\']+)\'/', $e->getAwsErrorMessage(), $matches);
                $correct_region = $matches[1] ?? 'eu-west-2';

                $s3 = new S3Client([
                    'version'     => 'latest',
                    'region'      => $correct_region,
                    'credentials' => [
                        'key'    => $creds->access_key,
                        'secret' => $creds->secret_key,
                    ],
                ]);

                try {
                    $result = $s3->listObjectsV2(['Bucket' => $bucket_name]);
                } catch (\Exception $e) {
                    $form['error'] = [
                        '#markup' => '<div class="messages messages--error">Unable to fetch files from region-adjusted bucket: ' . $e->getMessage() . '</div>',
                    ];
                    return $form;
                }
            } else {
                $form['error'] = [
                    '#markup' => '<div class="messages messages--error">Error fetching files: ' . $e->getMessage() . '</div>',
                ];
                return $form;
            }
        } catch (\Exception $e) {
            $form['error'] = [
                '#markup' => '<div class="messages messages--error">Error fetching files: ' . $e->getMessage() . '</div>',
            ];
            return $form;
        }

        $objects = $result['Contents'] ?? [];

        $form['s3_upload_section'] = [
            '#type'   => 'details',
            '#title'  => $this->t('Upload Files to S3'),
            '#open'   => true,
            '#weight' => -10,
        ];
        $form['s3_upload_section']['s3_upload_form'] = \Drupal::formBuilder()->getForm(S3FileUploadForm::class, $bucket_name);
        $form['grouped_tabs_wrapper']                = [
            '#type'       => 'container',
            '#attributes' => ['class' => ['s3-tab-wrapper', 'my-custom-tab-section']],
        ];

        $form['grouped_tabs_wrapper']['tabs'] = [
            '#type'       => 'vertical_tabs',
            '#title'      => $this->t('Grouped Files'),
            '#weight'     => 0,
            '#attributes' => ['class' => ['s3-vertical-tabs']],
        ];

        $image_exts = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $doc_exts   = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'csv'];
        $images     = [];
        $documents  = [];
        $others     = [];

        foreach ($objects as $object) {
            $key      = $object['Key'];
            $ext      = strtolower(pathinfo($key, PATHINFO_EXTENSION));
            $size_kb  = round($object['Size'] / 1024, 2);
            $modified = date('Y-m-d H:i:s', strtotime($object['LastModified']));

            $download_url = $s3->createPresignedRequest(
                $s3->getCommand('GetObject', ['Bucket' => $bucket_name, 'Key' => $key]),
                '+10 minutes'
            )->getUri();

            $preview_markup = in_array($ext, $image_exts)
            ? Markup::create('<img src="' . $download_url . '" class="w-100 s3-preview-td" style="max-height: 60px;" alt="' . htmlspecialchars($key) . '" />')
            : Markup::create('<span class="text-muted s3-preview-td">N/A</span>');

            $edit_url = Url::fromRoute('aws_manager.s3_file_edit_modal', [
                'bucket' => $bucket_name,
                'key'    => $key,
            ]);
            $edit_link = [
                '#type'       => 'link',
                '#title'      => $this->t('Edit'),
                '#url'        => $edit_url,
                '#attributes' => [
                    'class'                => [
                        'use-ajax',
                        'button',
                        'button--small',
                        'button--primary',
                        's3-edit-btn',
                        'dialog',
                        'aws-btn',
                        'aws-btn-save',
                    ],

                    'data-dialog-type'     => 'dialog',
                    'data-dialog-renderer' => 'off_canvas',
                    'data-dialog-options'  => Json::encode([
                        'width'       => 500,
                        'height'      => 400,
                        'dialogClass' => 'edit-s3-object-dialog',
                    ]),
                    'title'                => $this->t('Edit file'),
                ],
                '#attached'   => [
                    'library' => ['core/drupal.dialog.ajax'],
                ],
            ];

            $delete_url = Url::fromRoute('aws_manager.s3_file_delete', [
                'bucket' => $bucket_name,
                'key'    => $key,
            ]);
            $delete_link = [
                '#type'       => 'link',
                '#title'      => $this->t('Delete'),
                '#url'        => $delete_url,
                '#attributes' => [
                    'class'                => [
                        'use-ajax',
                        'button',
                        'button--small',
                        'button--danger',
                        's3-delete-btn',
                        'dialog',
                        'aws-btn',
                        'aws-btn-delete',
                    ],
                    'data-dialog-type'     => 'dialog',
                    'data-dialog-renderer' => 'off_canvas',
                    'data-dialog-options'  => Json::encode([
                        'width'       => 400,
                        'height'      => 200,
                        'dialogClass' => 'delete-s3-object-dialog',
                    ]),
                    'title'                => $this->t('Delete file'),
                ],
                '#attached'   => [
                    'library' => ['core/drupal.dialog.ajax'],
                ],
            ];

            $row = [
                'Preview'    => $preview_markup,
                'Key'        => ['data' => Markup::create(wordwrap(htmlspecialchars($key), 50, '<br>', true))],
                'Size'       => $size_kb . ' KB',
                'Modified'   => $modified,
                'Download'   => Link::fromTextAndUrl('Download', Url::fromUri($download_url))->toString(),
                'Operations' => [
                    'data' => [
                        '#type'       => 'container',
                        '#attributes' => ['class' => ['s3-ops-container', 'd-flex', 'gap-2']],
                        'edit'        => $edit_link,
                        'delete'      => $delete_link,
                    ],
                ],
            ];

            if (in_array($ext, $image_exts)) {
                $images[] = $row;
            } elseif (in_array($ext, $doc_exts)) {
                $documents[] = $row;
            } else {
                $others[] = $row;
            }
        }

        $form['grouped_tabs_wrapper']['image_files'] = [
            '#type'  => 'details',
            '#title' => $this->t('Image Files'),
            '#group' => 'tabs',
            '#open'  => true,
        ];
        $form['grouped_tabs_wrapper']['image_files']['table'] = [
            '#type'   => 'table',
            '#header' => ['Preview', 'File Key', 'Size (KB)', 'Last Modified', 'Download', 'Operations'],
            '#rows'   => $images,
            '#empty'  => $this->t('No image files found.'),
        ];

        $form['grouped_tabs_wrapper']['document_files'] = [
            '#type'  => 'details',
            '#title' => $this->t('Document Files'),
            '#group' => 'tabs',
        ];
        $form['grouped_tabs_wrapper']['document_files']['table'] = [
            '#type'   => 'table',
            '#header' => ['Preview', 'File Key', 'Size (KB)', 'Last Modified', 'Download', 'Operations'],
            '#rows'   => $documents,
            '#empty'  => $this->t('No document files found.'),
        ];

        $form['grouped_tabs_wrapper']['other_files'] = [
            '#type'  => 'details',
            '#title' => $this->t('Other Files'),
            '#group' => 'tabs',
        ];
        $form['grouped_tabs_wrapper']['other_files']['table'] = [
            '#type'   => 'table',
            '#header' => ['Preview', 'File Key', 'Size (KB)', 'Last Modified', 'Download', 'Operations'],
            '#rows'   => $others,
            '#empty'  => $this->t('No other files found.'),
        ];

        $form['list_page'] = [
            '#type'       => 'link',
            '#title'      => $this->t('← View All Buckets'),
            '#url'        => Url::fromRoute('aws_manager.aws_live_list'),
            '#attributes' => ['class' => ['btn', 'btn-link']],
        ];

        return $form;
    }

    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        // No submission logic
    }

}
