<?php

namespace Drupal\aws_manager\Form;

use Aws\S3\S3Client;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

class S3FileDeleteConfirmForm extends ConfirmFormBase {

  protected $bucket;
  protected $key;

  public function getFormId(): string {
    return 's3_file_delete_confirm_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $bucket = NULL, $key = NULL): array {
    $this->bucket = $bucket;
    $this->key = $key;

    $form['#prefix'] = '<div id="s3-delete-form-wrapper">';
    $form['#suffix'] = '</div>';

    $form = parent::buildForm($form, $form_state);

    $form['actions']['submit']['#ajax'] = [
      'callback' => '::ajaxSubmitHandler',
      'wrapper' => 's3-delete-form-wrapper',
      'progress' => [
        'type' => 'throbber',
        'message' => $this->t('Deleting...'),
      ],
    ];

    return $form;
  }

  public function getQuestion(): string {
    return $this->t("Are you sure you want to delete the file '@file'?", ['@file' => $this->key]);
  }

  public function getCancelUrl(): Url {
    return Url::fromRoute('<current>');
  }

  public function getConfirmText(): string {
    return $this->t('Delete');
  }

  public function ajaxSubmitHandler(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    $uid = $this->currentUser()->id();
    $db = \Drupal::database();
    $creds = $db->select('aws_credentials', 'c')
      ->fields('c')
      ->condition('uid', $uid)
      ->execute()
      ->fetchObject();

    if ($creds) {
      try {
        $s3 = new S3Client([
          'version' => 'latest',
          'region' => 'us-east-1',
          'credentials' => [
            'key' => $creds->access_key,
            'secret' => $creds->secret_key,
          ],
        ]);

        $s3->deleteObject([
          'Bucket' => $this->bucket,
          'Key' => $this->key,
        ]);

        $this->messenger()->addStatus($this->t("File '@file' deleted.", ['@file' => $this->key]));

        $redirect_path = "/aws/s3/{$this->bucket}/browse";
        $response->addCommand(new RedirectCommand($redirect_path));
      }
      catch (\Exception $e) {
        $this->messenger()->addError($this->t('Delete failed: @msg', ['@msg' => $e->getMessage()]));
      }
    }
    else {
      $this->messenger()->addError($this->t('AWS credentials not found.'));
    }

    return $response;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Empty because deletion handled via AJAX.
  }

}
