<?php

namespace Drupal\azure_ai_services\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Azure Translator settings for this site.
 */
class AzureTranslatorSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'azure_translator_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'azure_openai.translate',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('azure_openai.translate');

    $stored_key = $config->get('translator_api_key');
    $masked_placeholder = $stored_key ? str_repeat('*', strlen($stored_key)) : '';

    $form['translator_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Translator Endpoint'),
      '#description' => $this->t('Usually https://api.cognitive.microsofttranslator.com'),
      '#default_value' => $config->get('translator_endpoint') ?? '',
      '#required' => TRUE,
    ];

    $form['translator_region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Translator Region'),
      '#description' => $this->t('Azure region where your Translator resource is located (e.g., canadaeast).'),
      '#default_value' => $config->get('translator_region') ?? '',
      '#required' => TRUE,
    ];

    $form['translator_api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Translator API Key'),
      '#description' => $stored_key ? $this->t('Enter a new key to replace the existing one. Leave blank to keep the current key.') : '',
      '#default_value' => '',
      '#required' => !$stored_key,
      '#attributes' => [
        'placeholder' => $masked_placeholder,
        'autocomplete' => 'off',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('azure_openai.translate')
      ->set('translator_endpoint', $form_state->getValue('translator_endpoint'))
      ->set('translator_region', $form_state->getValue('translator_region'))
      ->save();

    $new_key = $form_state->getValue('translator_api_key');
    if (!empty($new_key)) {
      $this->config('azure_openai.translate')
        ->set('translator_api_key', $new_key)
        ->save();
    }
  }

}
