<?php

namespace Drupal\azure_ai_services\Service;

use GuzzleHttp\Client;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for Azure Translator Text API.
 */
class AzureTranslatorService {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $client;

  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
    $this->client = new Client();
  }

  /**
   * Translate text using Azure Translator Text API.
   *
   * @param string $text
   *   The text to translate.
   * @param string $to
   *   Target language (e.g., 'fr').
   * @param string|null $from
   *   Source language (optional, e.g., 'en').
   *
   * @return string|null
   *   The translated text or NULL on failure.
   */
  public function translateText(string $text, string $to = 'fr', ?string $from = NULL): ?string {
    $config = $this->configFactory->get('azure_openai.translate');
    $apiKey = $config->get('translator_api_key');
    $region = $config->get('translator_region');
    $base_endpoint = $config->get('translator_endpoint') ?? '';
    $endpoint = rtrim((string) $base_endpoint, '/') . '/translate';

    $query = [
      'to' => $to,
      'from' => $from,
      'api-version' => '3.0',
      'textType' => 'html',
    ];

    try {
      $response = $this->client->post($endpoint, [
        'headers' => [
          'Ocp-Apim-Subscription-Key' => $apiKey,
          'Ocp-Apim-Subscription-Region' => $region,
          'Content-Type' => 'application/json',
        ],
        'query' => $query,
        'json' => [
          ['Text' => $text],
        ],
        'verify' => FALSE,
      ]);

      $body = json_decode($response->getBody(), TRUE);

      return $body[0]['translations'][0]['text'] ?? NULL;
    }
    catch (\Exception $e) {
      \Drupal::logger('azure_openai')->error('Translation failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

}
