<?php

declare(strict_types=1);

namespace Drupal\Tests\b24\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Delete;
use Drupal\Core\Database\Query\Select;
use Drupal\Core\Database\Query\Update;
use Drupal\Tests\UnitTestCase;
use Drupal\b24\Service\ReferenceManager;
use Drupal\sqlite\Driver\Database\sqlite\Statement;
use Drupal\user\Entity\User;

/**
 * Tests for ReferenceManager service.
 *
 * @coversDefaultClass \Drupal\b24\Service\ReferenceManager
 * @group b24
 */
final class ReferenceManagerTest extends UnitTestCase {

  /**
   * The b24 reference manager.
   *
   * @var \Drupal\b24\Service\ReferenceManager
   */
  protected ReferenceManager $referenceManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $mockConnection = $this->createMock(Connection::class);
    $mockDelete = $this->getMockBuilder(Delete::class)
      ->disableOriginalConstructor()
      ->getMock();
    $mockDelete
      ->method('execute')
      ->willReturn(1);
    $mockDelete
      ->method('condition')
      ->willReturnSelf();

    $mockSelectStatement = $this->getMockBuilder(Statement::class)
      ->disableOriginalConstructor()
      ->getMock();
    $mockSelectStatement->expects($this->any())
      ->method('fetchAssoc')
      ->willReturn(['ext_id' => 1, 'hash' => 'sdf234jhgf']);
    $mockSelect = $this->getMockBuilder(Select::class)
      ->disableOriginalConstructor()
      ->getMock();
    $mockSelect
      ->method('condition')
      ->willReturnSelf();
    $mockSelect
      ->method('fields')
      ->willReturnSelf();
    $mockSelect
      ->method('execute')
      ->willReturn($mockSelectStatement);

    $mockInsert = $this->getMockBuilder(Update::class)
      ->disableOriginalConstructor()
      ->getMock();
    $mockInsert
      ->method('fields')
      ->willReturnSelf();
    $mockInsert
      ->method('execute')
      ->willReturn(1);

    $mockUpdate = $this->getMockBuilder(Update::class)
      ->disableOriginalConstructor()
      ->getMock();
    $mockUpdate
      ->method('condition')
      ->willReturnSelf();
    $mockUpdate
      ->method('fields')
      ->willReturnSelf();
    $mockUpdate
      ->method('execute')
      ->willReturn(1);

    $mockConnection->method('delete')
      ->willReturn($mockDelete);

    $mockConnection->method('select')
      ->willReturn($mockSelect);

    $mockConnection->method('update')
      ->willReturn($mockUpdate);
    $this->referenceManager = new ReferenceManager($mockConnection);
  }

  /**
   * @covers ::getReference
   */
  public function testGetReference() {
    $user = $this->getMockBuilder(User::class)
      ->disableOriginalConstructor()
      ->getMock();
    $result = $this->referenceManager->getReference($user, 'contact');
    $this->assertIsArray($result);
    $this->assertContains('sdf234jhgf', $result);
  }

  /**
   * @covers ::deleteReference
   */
  public function testDeleteReference() {
    $result = $this->referenceManager->deleteReference('contact', 1);
    $this->assertTrue($result > 0);
  }

  /**
   * @covers ::getHash
   */
  public function testGetHash() {
    $fields = [
      'one_field_key' => 'one_field_value',
      'another_field_key' => 'another_field_value',
    ];
    $expected = 'u8wL2k9KsIRyxfPPO4wsXKKq3lI7MCMZprnLqZLtbPE';
    $hash = $this->referenceManager->getHash($fields);

    $this->assertEquals($expected, $hash);
  }

  /**
   * @covers ::updateHash
   */
  public function testUpdateHash() {
    $entity = $this->getMockBuilder(User::class)
      ->disableOriginalConstructor()
      ->getMock();
    $result = $this->referenceManager->updateHash($entity, 'contact', 'qwerty');
    $this->assertIsInt($result);
  }

}
