<?php

namespace Drupal\b24_commerce\EventSubscriber;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\b24\B24Interface;
use Drupal\b24\Service\ReferenceManager;
use Drupal\b24\Service\RestManager;
use Drupal\b24_commerce\Event\B24CommerceEvent;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Event\OrderAssignEvent;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * B24_commerce event subscriber.
 */
class B24CommerceSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The Bitrix24 rest manager service.
   *
   * @var \Drupal\b24\Service\RestManager
   */
  protected $restManager;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The user account interface.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $dispatcher;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The B24 reference manager.
   *
   * @var \Drupal\b24\Service\ReferenceManager
   */
  protected $referenceManager;

  /**
   * The CRM mode (classic or simple).
   *
   * @var string
   */
  protected string $crmMode;

  /**
   * The Bitrix 24 entity type ID we operate on.
   *
   * @var string
   */
  protected string $operatedB24EntityType;

  /**
   * Constructs event subscriber.
   */
  public function __construct(
    MessengerInterface $messenger,
    RestManager $b24_rest_manager,
    ConfigFactory $config_factory,
    Token $token,
    Connection $connection,
    AccountInterface $account,
    EventDispatcherInterface $event_dispatcher,
    ModuleHandler $module_handler,
    ReferenceManager $reference_manager,
  ) {
    $this->messenger = $messenger;
    $this->restManager = $b24_rest_manager;
    $this->configFactory = $config_factory;
    $this->token = $token;
    $this->database = $connection;
    $this->currentUser = $account;
    $this->dispatcher = $event_dispatcher;
    $this->moduleHandler = $module_handler;
    $this->referenceManager = $reference_manager;
    $this->crmMode = $this->restManager->getCrmMode();
    $this->operatedB24EntityType = $this->crmMode == B24Interface::CRM_MODE_CLASSIC ? 'lead' : 'deal';
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      OrderEvents::ORDER_INSERT => 'onOrderInsert',
      OrderEvents::ORDER_UPDATE => 'onOrderUpdate',
      OrderEvents::ORDER_ASSIGN => 'onOrderAssign',
      OrderEvents::ORDER_DELETE => 'onOrderDelete',
    ];
  }

  /**
   * The function executed on commerce order insert event.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The commerce order insert event.
   */
  public function onOrderInsert(OrderEvent $event):void {
    $order = $event->getOrder();
    $fields = $this->getValues($order, NULL, $this->operatedB24EntityType);
    if (!$fields) {
      return;
    }
    $context = [
      'op' => 'insert',
      'entity_name' => $this->operatedB24EntityType,
      'order' => $order,
      'mode' => $this->crmMode,
    ];
    $this->moduleHandler->alter('b24_commerce_data', $fields, $context);
    $ext_id = $this->operatedB24EntityType == 'lead' ? $this->restManager->addLead($fields) : $this->restManager->addDeal($fields);
    $event = new B24CommerceEvent($order, $this->operatedB24EntityType, $ext_id);
    $this->dispatcher->dispatch($event, B24CommerceEvent::ENTITY_INSERT);
    if ($ext_id) {
      $hash = $this->referenceManager->getHash($fields);
      $this->referenceManager->addReference($order->id(), 'commerce_order', $ext_id, $this->operatedB24EntityType, $hash);
    }
    $this->updateProducts($order, $this->operatedB24EntityType);
  }

  /**
   * The function executed on commerce order update event.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The commerce order event.
   */
  public function onOrderUpdate(OrderEvent $event): void {
    $order = $event->getOrder();
    $this->updateOrder($order);
  }

  /**
   * The function executed on commerce order assign event.
   *
   * @param \Drupal\commerce_order\Event\OrderAssignEvent $event
   *   The commerce order assign event.
   */
  public function onOrderAssign(OrderAssignEvent $event): void {
    $mail = $this->currentUser->getEmail();
    $order = $event->getOrder();
    $order->setEmail($mail);
    $this->updateOrder($order);
  }

  /**
   * Callback executed on commerce order delete event.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The commerce order event.
   */
  public function onOrderDelete(OrderEvent $event): void {
    $order = $event->getOrder();
    $reference = $this->referenceManager->getReference($order, $this->operatedB24EntityType);
    if (!empty($reference['ext_id'])) {
      $this->referenceManager->deleteReference($this->operatedB24EntityType, $order->id(), 'commerce_order');
    }
  }

  /**
   * Collects field values for a chosen Bitrix24 entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   Drupal commerce order.
   * @param null|int $id
   *   Bitrix24 entity id.
   * @param string $entity
   *   Bitrix24 entity type id.
   *
   * @return array
   *   An array of fields.
   */
  public function getValues(OrderInterface $order, ?int $id = NULL, string $entity = 'lead'): array {
    $config = $this->configFactory->get("b24_commerce.mapping.{$order->bundle()}");
    if (!$config->get($entity)) {
      return [];
    }
    $fields = array_filter($config->get($entity));
    $profile = $order->getBillingProfile();
    $user = $order->getCustomer();
    $field_definitions = $this->configFactory->get("b24_commerce.field_types")->get('definition');
    if ($id) {
      $method_name = 'get' . ucfirst($entity);
      $existing_entity = $this->restManager->{$method_name}($id);
    }
    foreach ($fields as $field_name => &$value) {
      if (!isset($field_definitions[$field_name])) {
        continue;
      }
      $bubbleable_metadata = new BubbleableMetadata();
      $value = nl2br($this->token->replace($value,
        ['commerce_order' => $order, 'profile' => $profile, 'user' => $user],
        ['clear' => TRUE], $bubbleable_metadata));
      if ($field_definitions[$field_name]['isRequired'] && !$value) {
        $value = $this->t('- Not set -');
      }
      /*
       * In Bitrix24 each item of a multiple field is a separate entity.
       * Updating multiple field items without mentioning id causes adding
       * another item. So we need to get all the related ids previously.
       */
      if ($field_definitions[$field_name]['type'] == 'crm_multifield') {
        $value = [['VALUE' => $value, 'VALUE_TYPE' => 'HOME']];
        if (!empty($existing_entity[$field_name])) {
          // @todo Support multiple drupal fields.
          $value[0]['ID'] = reset($existing_entity[$field_name])['ID'];
        }
      }
    }
    return $fields;
  }

  /**
   * Updates an existing Bitrix24 order.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The commerce order.
   *
   * @return array|bool
   *   Result of update.
   */
  public function updateOrder(OrderInterface $order): array|bool {
    $reference = $this->referenceManager->getReference($order, $this->operatedB24EntityType);
    $update = FALSE;
    if (!empty($reference['ext_id'])) {
      $fields = $this->getValues($order, $reference['ext_id'], $this->operatedB24EntityType);
      if (!$fields) {
        return FALSE;
      }
      $hash = $this->referenceManager->getHash($fields);
      if ($hash !== $reference['hash']) {
        if ($this->crmMode == B24Interface::CRM_MODE_CLASSIC) {
          $context = [
            'op' => 'update',
            'entity_name' => $this->operatedB24EntityType,
            'order' => $order,
            'mode' => $this->crmMode,
          ];
          $this->moduleHandler->alter('b24_commerce_data', $fields, $context);
          $update = $this->restManager->updateLead($reference['ext_id'], $fields);
        }
        else {
          // Get contacts created by this lead.
          $search = $this->restManager->get('crm.contact.list', ['filter' => ['DEAL_ID' => $reference['ext_id']]]);
          $contact = reset($search['result']);
          $context = [
            'op' => 'update',
            'entity_name' => 'contact',
            'order' => $order,
            'mode' => $this->crmMode,
          ];
          $this->moduleHandler->alter('b24_commerce_data', $fields, $context);
          $update = $this->restManager->updateContact($contact['ID'], $fields);
        }
        $this->referenceManager->updateHash($order, $this->operatedB24EntityType, $hash);
      }
    }
    $this->updateProducts($order, $this->operatedB24EntityType);
    $event = new B24CommerceEvent($order, $this->operatedB24EntityType, $update);
    $this->dispatcher->dispatch($event, B24CommerceEvent::ENTITY_UPDATE);
    return $update;
  }

  /**
   * Generates an array of bought products to be exported to Bitrix24.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The commerce order.
   *
   * @return array
   *   An array of product rows.
   */
  public function getProductRows(OrderInterface $order): array {
    $order_items = $order->getItems();
    $product_rows = [];
    foreach ($order_items as $order_item) {
      /** @var \Drupal\commerce_order\Entity\OrderItem $order_item */
      if (!$order_item->getPurchasedEntity()) {
        continue;
      }
      $product_rows[$order_item->getPurchasedEntity()->id()] = [
        'quantity' => $order_item->getQuantity(),
        'price' => $order_item->getPurchasedEntity()->getPrice()->getNumber(),
      ];
    }

    $rows = [];
    $product_ids = array_keys($product_rows);
    if ($product_ids) {
      $ext_products = $this->restManager->getList('product', ['filter' => ['XML_ID' => $product_ids]]);
      foreach ($ext_products as $ext_product) {
        $rows[] = [
          'PRODUCT_ID' => $ext_product['ID'],
          'PRICE' => $product_rows[$ext_product['XML_ID']]['price'],
          'QUANTITY' => $product_rows[$ext_product['XML_ID']]['quantity'],
        ];
      }
    }
    return $rows;
  }

  /**
   * Updates products for deal/lead.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The commerce order.
   * @param string $entity
   *   The bitrix24 entity type id.
   *
   * @return bool
   *   The result of update.
   */
  public function updateProducts(OrderInterface $order, string $entity = 'lead'): bool {
    $rows = $this->getProductRows($order);
    $reference = $this->referenceManager->getReference($order, $entity);
    if (!$reference) {
      return FALSE;
    }

    if ($entity == 'lead') {
      $result = $this->restManager->setLeadProducts($reference['ext_id'], $rows);
    }
    elseif ($entity == 'deal') {
      $result = $this->restManager->setDealProducts($reference['ext_id'], $rows);
    }

    return $result ?? FALSE;
  }

}
