<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\StorageCacheInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\TypedData\TraversableTypedDataInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Helper service for the 'config' plugin.
 */
class BabelConfigHelper implements BabelConfigHelperInterface {

  public function __construct(
    protected readonly StorageCacheInterface $storage,
    protected readonly TypedConfigManagerInterface $typedConfigManager,
    #[Autowire(service: 'babel.backend_chained_cache')]
    protected readonly CacheBackendInterface $cache,
  ) {}

  /**
   * Returns a list of paths to the translatable properties of a config object.
   *
   * @param string $name
   *   The configuration name.
   *
   * @return array<array-key, string>
   *   Associative array where the key is the path to the translatable string,
   *   as a string with dot as separator, and the value is the translation
   *   context.
   */
  public function getTranslatableProperties(string $name): array {
    $cid = 'babel.config_translatables';

    $cache = $this->cache->get($cid);
    if ($cache === FALSE) {
      $translatables = [];
    }
    else {
      $translatables = $cache->data ?: [];
      if (isset($translatables[$name])) {
        return $translatables[$name];
      }
    }

    $paths = [];

    if ($data = $this->storage->read($name)) {
      $this->doGetTranslatableStrings($paths, $name, $data);
      $translatables[$name] = $paths;
      $this->cache->set($cid, $translatables);
    }

    return $paths;
  }

  /**
   * Processes translatable data within a nested data structure.
   *
   * @param array<array-key, string> $paths
   *   Passed by reference and populated during recursion. Associative array
   *   where the key is the path to the translatable string, as a string with
   *   dot as separator, and the value is the translation context.
   * @param string $name
   *   The configuration object name.
   * @param array $data
   *   The configuration data as an array.
   * @param \Drupal\Core\TypedData\TypedDataInterface|null $element
   *   Typed sata element. Used internally.
   * @param array $path
   *   Path to one translatable property. Used internally.
   */
  protected function doGetTranslatableStrings(array &$paths, string $name, mixed $data, ?TypedDataInterface $element = NULL, array $path = []): void {
    $element ??= $this->typedConfigManager->createFromNameAndData($name, $data);

    if ($element instanceof TraversableTypedDataInterface) {
      foreach ($element as $key => $childElement) {
        // This is a mapping/sequence, descent to a deeper level.
        $this->doGetTranslatableStrings($paths, $name, $data, $childElement, [...$path, $key]);
      }
    }
    else {
      $string = $element->getValue();
      // Process if the value is a non-empty string.
      if (is_string($string) && trim($string)) {
        $definition = $element->getDataDefinition();
        if (!empty($definition['translatable'])) {
          $context = $definition['translation context'] ?? '';
          $paths[implode('.', $path)] = $context;
        }
      }
    }
  }

}
