<?php

declare(strict_types=1);

namespace Drupal\babel\EventSubscriber;

use Drupal\babel\BabelConfigHelperInterface;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\StorageCacheInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Database\DatabaseExceptionWrapper;
use Drupal\Core\DestructableInterface;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to configuration CRUD.
 */
class ConfigSubscriber implements EventSubscriberInterface, DestructableInterface {

  /**
   * List of affected configuration names.
   *
   * @var string[]
   */
  protected array $names = [];

  public function __construct(
    public readonly StorageCacheInterface $storage,
    public readonly TypedConfigManagerInterface $typedConfigManager,
    public readonly BabelStorageInterface $babelStorage,
    public readonly BabelConfigHelperInterface $configHelper,
    #[Autowire(service: 'language_manager')]
    public readonly ConfigurableLanguageManagerInterface $languageManager,
    #[Autowire(service: 'babel.backend_chained_cache')]
    protected readonly CacheBackendInterface $cache,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConfigEvents::DELETE => 'onConfigDelete',
      ConfigEvents::SAVE => [['onConfigChange'], ['onConfigUpdate']],
    ];
  }

  /**
   * Cleans the {babel_source_instance} table after a config has been deleted.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The event.
   */
  public function onConfigDelete(ConfigCrudEvent $event): void {
    $this->babelStorage->delete('config', $event->getConfig()->getName() . ':');
  }

  /**
   * Updates the {babel_source_instance} table after a config is added.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The event.
   */
  public function onConfigChange(ConfigCrudEvent $event): void {
    $name = $event->getConfig()->getName();
    $this->names[] = $name;

    // Clear the 'babel.config_translatables' cache for this entry.
    if ($cache = $this->cache->get('babel.config_translatables')) {
      $data = $cache->data;
      if (is_array($data)) {
        unset($data[$name]);
        $this->cache->set('babel.config_translatables', $data);
      }
    }
  }

  /**
   * Removes translations if a source string has changed.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The event.
   */
  public function onConfigUpdate(ConfigCrudEvent $event): void {
    if ($event->getConfig()->isNew()) {
      return;
    }

    $configName = $event->getConfig()->getName();

    $originalData = $event->getConfig()->getOriginal();
    $currentData = $event->getConfig()->get();

    foreach ($this->configHelper->getTranslatableProperties($configName) as $path => $context) {
      $pathArray = explode('.', $path);
      $originalValue = NestedArray::getValue($originalData, $pathArray);
      $currentValue = NestedArray::getValue($currentData, $pathArray);

      if ($originalValue !== $currentValue) {
        try {
          // Remove the source string from babel_source_instance table.
          $this->babelStorage->delete('config', "$configName:$path");

          // Remove translated config strings from all language overrides.
          $this->removeConfigTranslations($configName, $path);
        }
        catch (DatabaseExceptionWrapper) {
          // During module installation or testing,
          // the babel_source_instance table may not exist yet.
        }
      }
    }
  }

  /**
   * Removes translated config strings from all language overrides.
   *
   * @param string $configName
   *   The configuration name.
   * @param string $path
   *   The configuration path.
   */
  protected function removeConfigTranslations(string $configName, string $path): void {
    $languages = $this->languageManager->getLanguages();
    $defaultLanguage = $this->languageManager->getDefaultLanguage()->getId();

    foreach ($languages as $langcode => $language) {
      if ($langcode === $defaultLanguage) {
        continue;
      }

      $override = $this->languageManager->getLanguageConfigOverride($langcode, $configName);
      if ($override->get($path) !== NULL) {
        $override->clear($path);
        $override->save();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function destruct(): void {
    if (!$this->names) {
      return;
    }

    $sources = [];
    foreach ($this->names as $name) {
      $paths = $this->configHelper->getTranslatableProperties($name);
      $data = $this->storage->read($name);
      foreach ($paths as $path => $context) {
        $sources["$name:$path"] = new Source(
          string: NestedArray::getValue($data, explode('.', $path)),
          context: $context,
        );
      }
      try {
        $this->babelStorage->delete('config', "$name:");
      }
      catch (DatabaseExceptionWrapper) {
        // On module uninstall process, the table has been already removed (here
        // we run on kernel termination).
      }
    }

    $this->babelStorage->update('config', $sources);

    // Strings were processed, make sure it starts on clean on a new call.
    $this->names = [];
  }

}
