<?php

declare(strict_types=1);

namespace Drupal\babel\Model;

use Drupal\Component\Gettext\PoItem;

/**
 * Representation of a translatable string.
 */
class StringTranslation {

  /**
   * Value object representing a translated or untranslated string.
   *
   * @param \Drupal\babel\Model\Source $source
   *   The source string object.
   * @param \Drupal\babel\Model\Translation|null $translation
   *   The translation string object.
   */
  public function __construct(
    public readonly Source $source,
    protected ?Translation $translation = NULL,
  ) {}

  /**
   * Sets the translation.
   *
   * @param \Drupal\babel\Model\Translation $translation
   *   The translation to be set.
   *
   * @return $this
   */
  public function setTranslation(Translation $translation): self {
    $this->translation = $translation;
    return $this;
  }

  /**
   * Returns the string translation.
   *
   * @return \Drupal\babel\Model\Translation|null
   *   The current translation.
   */
  public function getTranslation(): ?Translation {
    return $this->translation;
  }

  /**
   * Checks whether the source string is translated.
   *
   * @return bool
   *   Whether the source string is translated.
   */
  public function isTranslated(): bool {
    return $this->translation instanceof Translation && trim($this->translation->string);
  }

  /**
   * Checks whether the string represents a plural string.
   *
   * @return bool
   *   Whether the string represents a plural string.
   */
  public function isPlural(): bool {
    return count($this->getSourcePluralVariants()) > 1;
  }

  /**
   * Returns the source string plural variants.
   *
   * @return list<string>
   *   The source-string plural variants.
   */
  public function getSourcePluralVariants(): array {
    return explode(PoItem::DELIMITER, $this->source->string);
  }

  /**
   * Returns the translated string plural variants.
   *
   * @return list<string>
   *   The translated string plural variants.
   */
  public function getTranslatedPluralVariants(): array {
    return $this->translation ? explode(PoItem::DELIMITER, (string) $this->translation->string) : [];
  }

  /**
   * Returns an array representation of the object.
   *
   * @return array{source: string, translation: ?string}
   *   Array representation of the object.
   */
  public function toArray(): array {
    return [
      'source' => $this->source,
      'translation' => $this->translation,
    ];
  }

}
