<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel\TranslationType;

use Drupal\babel\BabelConfigHelperInterface;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\TranslationType;
use Drupal\babel\Plugin\Babel\TranslationTypePluginBase;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageCacheInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles translatable strings from configuration objects.
 */
#[TranslationType(
  id: 'config',
  label: new TranslatableMarkup('Configuration'),
)]
class Config extends TranslationTypePluginBase {

  public function __construct(
    array $configuration,
    string $pluginId,
    array $pluginDefinition,
    BabelStorageInterface $babelStorage,
    protected readonly StorageCacheInterface $storage,
    protected readonly ConfigurableLanguageManagerInterface $languageManager,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly BabelConfigHelperInterface $babelConfigHelper,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $babelStorage);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(BabelStorageInterface::class),
      $container->get('config.storage'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get(BabelConfigHelperInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getStrings(string $langcode, array $ids = []): array {
    // @todo Consider replacing and simplifying this with high level API calls
    //   in https://drupal.org/i/3535124.
    if (!$ids) {
      $ids = $this->babelStorage->getBaseQuery($this->getPluginId(), fields: ['id'])->execute()->fetchAllKeyed(0, 0);
    }

    $list = $pathsByName = [];
    foreach ($ids as $id) {
      [$name, $path] = explode(':', $id, 2);
      $pathsByName[$name][] = explode('.', $path);
    }

    $names = array_keys($pathsByName);
    $sources = $this->storage->readMultiple($names);
    $storage = $this->storage->createCollection("language.$langcode");
    $translations = $storage->readMultiple($names) ?: [];

    foreach ($pathsByName as $name => $paths) {
      $translatables = $this->babelConfigHelper->getTranslatableProperties($name);
      foreach ($paths as $path) {
        $configTranslations = $translations[$name] ?? [];
        $translation = NestedArray::getValue($configTranslations, $path);
        $pathAsString = implode('.', $path);

        $list["$name:$pathAsString"] = new StringTranslation(
          source: new Source(
            string: NestedArray::getValue($sources[$name], $path),
            context: $translatables[$pathAsString],
          ),
          translation: $translation ? new Translation(
            language: $langcode,
            string: $translation,
          ) : NULL,
        );
      }
    }

    return $list;
  }

  /**
   * {@inheritdoc}
   */
  public function getString(string $langcode, string $id): ?StringTranslation {
    // @todo Consider replacing and simplifying this with high level API calls
    //   in https://drupal.org/i/3535124.
    $row = $this->babelStorage->getBaseQuery($this->getPluginId(), fields: ['id'], includeSortKey: FALSE)
      ->condition('id', $id)
      ->execute()
      ->fetch();

    if (!$row) {
      return NULL;
    }

    [$name, $pathAsString] = explode(':', $id, 2);
    $data = $this->storage->read($name);
    if (!$data) {
      return NULL;
    }

    $path = explode('.', $pathAsString);
    if (!$sourceString = NestedArray::getValue($data, $path)) {
      return NULL;
    }

    $storage = $this->storage->createCollection("language.$langcode");
    $translation = $storage->read($name);

    $translatables = $this->babelConfigHelper->getTranslatableProperties($name);
    $propertyTranslation = $translation ? NestedArray::getValue($translation, $path) : NULL;

    return new StringTranslation(
      source: new Source(
        string: $sourceString,
        context: $translatables[$pathAsString],
        status: (bool) $row->status,
      ),
      translation: $propertyTranslation ? new Translation(
        language: $langcode,
        string: $propertyTranslation,
      ) : NULL,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function updateTranslation(StringTranslation $string, string $id, string $langcode, string $translation): void {
    [$name, $path] = explode(':', $id, 2);
    $override = $this->languageManager->getLanguageConfigOverride($langcode, $name);
    if ($translation) {
      $override->set($path, $translation);
    }
    else {
      $override->clear($path);
    }
    $override->save();
    $this->configFactory->reset();

    parent::updateTranslation($string, $id, $langcode, $translation);
  }

}
