<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Behat\Mink\Element\DocumentElement;
use Behat\Mink\Element\NodeElement;
use Drupal\babel\Model\Source;
use Drupal\Core\Url;
use Drupal\locale\StringStorageInterface;
use Drupal\locale\TranslationString;

/**
 * Tests how translations are imported and exported.
 *
 * @group babel
 */
class ExportImportTest extends BabelFunctionalJavascriptTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Warm cache by visiting the login page with the configured test language
    // interface.
    $url = Url::fromRoute(
      route_name: 'user.login',
      options: ['language' => $this->language],
    );
    $this->drupalGet($url->toString());
  }

  /**
   * Test export to CSV and import from CSV.
   */
  public function testExportImport(): void {
    $csvExport = $this->assertCsvExport();

    $sources = [
      'Account cancellation request for [user:display-name] at [site:name]',
      'Account details for [user:display-name] at [site:name] (approved)',
      'Account details for [user:display-name] at [site:name] (blocked)',
      'Account details for [user:display-name] at [site:name] (canceled)',
      'Account details for [user:display-name] at [site:name] (pending admin approval)',
      'Account details for [user:display-name] at [site:name]',
      'Administration',
      'Administrative task links',
      'An administrator created an account for you at [site:name]',
      'An AJAX HTTP error occurred.',
      'An AJAX HTTP request terminated abnormally.',
      'An error occurred during the execution of the Ajax response: !error',
    ];

    $csv = ['"Source string","Translated string","Context","Notes","URL","ID (do not edit)"'];
    foreach ($sources as $source) {
      $csv[] = '"' . $source . '","","","","","' . (new Source($source, ''))->getHash() . '"';
    }

    // Check locale translations line by line, without their ID if the locale
    // translation identifiers do not prove to be stable enough.
    $this->assertStringContainsString(implode("\n", $csv), $csvExport);

    // Add translation for "Log in" (locale text) and "Powered by.." (config)
    // strings.
    $csvToImport = preg_replace(
      [
        '/^("Log in",)("")/m',
        '/^("Powered by <a href="":poweredby"">Drupal<\/a>",)("")/m',
      ],
      [
        '$1"[TRANSLATED] Log in"',
        '$1"[TRANSLATED] Powered by <a href="":poweredby"">Drupal</a>"',
      ],
      $csvExport
    );

    $this->assertCsvImport(
      csvToImport: $csvToImport,
      translationsToVerify: [
        '[TRANSLATED] Log in',
        '[TRANSLATED] Powered by Drupal',
      ],
      routeToCheckForTranslatedStrings: 'user.login',
    );

    $this->assertTranslationDeletion(
      translationsToDelete: [
        '[TRANSLATED] Log in',
        '[TRANSLATED] Powered by <a href=":poweredby">Drupal</a>',
      ],
      routeToCheck: 'user.login',
    );
  }

  /**
   * Verifies that CSV export works as expected.
   *
   * @return string
   *   The actually exported CSV. This is needed for testing the import.
   */
  protected function assertCsvExport(): string {
    $assertSession = $this->assertSession();

    $this->drupalGet(Url::fromRoute('babel.export'));
    $assertSession->pageTextContains('You are not authorized to access this page.');

    $this->drupalLogin($this->testAccount);
    // This request triggers JavaScript translated strings to be discovered and
    // added by Babel's JS alter hook.
    $this->drupalGet(Url::fromRoute('babel.export'));

    $assertSession->waitForField('Language')
      ?->selectOption(static::LANGCODE);
    $assertSession->waitForField('Exporter')
      ?->selectOption('Spreadsheet');
    $assertSession->waitForField('Extension')
      ?->selectOption('.csv');
    $assertSession->waitForButton('Export')?->click();

    $assertSession->statusMessageContainsAfterWait(
      'translation file has been exported. Download the Spreadsheet file'
    );

    $fileLink = $assertSession->waitForLink('here');
    $this->assertInstanceOf(NodeElement::class, $fileLink);
    $fileUri = $this->trimFilePath($fileLink->getAttribute('href'));
    $fileContent = file_get_contents(DRUPAL_ROOT . '/' . $fileUri);
    $this->assertIsString($fileContent);

    $this->drupalLogout();

    return $fileContent;
  }

  /**
   * Verifies that translation import works as expected.
   *
   * @param string $csvToImport
   *   The CSV string to import.
   * @param list<string> $translationsToVerify
   *   The translated strings which should be present on the page.
   * @param string $routeToCheckForTranslatedStrings
   *   The route of the page where the translated strings appear.
   */
  protected function assertCsvImport(
    string $csvToImport,
    array $translationsToVerify,
    string $routeToCheckForTranslatedStrings,
  ): void {
    $assertSession = $this->assertSession();

    $this->drupalGet(Url::fromRoute('babel.import'));
    $assertSession->pageTextContains('You are not authorized to access this page.');

    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.import'));

    $assertSession->waitForField('Language')
      ?->selectOption(static::LANGCODE);
    $assertSession->waitForField('Importer')
      ?->selectOption('Spreadsheet');

    $tmpfilePath = \Drupal::service('file_system')->tempnam('temporary://', 'test_') . '.csv';
    file_put_contents($tmpfilePath, $csvToImport);
    $assertSession->waitForField('File to import')
      ?->attachFile($this->container->get('file_system')->realpath($tmpfilePath));

    $this->assertNotEmpty($assertSession->waitForButton('Remove'));
    $this->getSession()->getPage()->findButton('Import')->click();

    $assertSession->statusMessageContainsAfterWait(
      sprintf('%s items imported to the %s translation from the %s Spreadsheet file.',
        count($translationsToVerify),
        \Drupal::languageManager()->getLanguage(static::LANGCODE)->getName(),
        basename($tmpfilePath),
      )
    );

    $this->drupalLogout();

    $path = Url::fromRoute(
      route_name: $routeToCheckForTranslatedStrings,
      options: ['language' => $this->language],
    )->toString();

    $this->assertTextOnPage($path, $translationsToVerify);
  }

  /**
   * Verifies that deleting translations works as expected.
   *
   * Also, implicitly tests Babel's JS alter hook.
   *
   * @param array $translationsToDelete
   *   Translation strings to delete.
   * @param string $routeToCheck
   *   The route which should not contain the deleted strings after the
   *   operation.
   */
  protected function assertTranslationDeletion(
    array $translationsToDelete,
    string $routeToCheck,
  ): void {
    // Ensure current user is logged out.
    if ($this->loggedInUser) {
      $this->drupalLogout();
    }

    $path = Url::fromRoute(
      route_name: $routeToCheck,
      options: ['language' => $this->language],
    )->toString();
    // Initially, text must be present.
    $this->assertTextOnPage(
      $path,
      // Test might contain HTML, let's remove the tags!
      // (But let's be aware of that this won't have the expected result if some
      // alternative text is present, e.g. img tags with alt attribute.)
      array_map(
        fn (string $translation) => strip_tags($translation),
        $translationsToDelete,
      ),
    );

    // Delete the translated strings.
    $localeStorage = \Drupal::service('locale.storage');
    $this->assertInstanceOf(StringStorageInterface::class, $localeStorage);
    $deleted = [];
    foreach ($translationsToDelete as $translation) {
      $translationString = $localeStorage->findTranslation([
        'translation' => $translation,
        'translated' => TRUE,
        'language' => static::LANGCODE,
      ]);
      $this->assertInstanceOf(TranslationString::class, $translationString);
      $deleted[] = $translationString->lid;
      $translationString->delete();
    }

    // Clear cache and force refresh of JavaScript translations.
    // After this, the next request triggers JavaScript translated strings to be
    // removed by Babel's JS alter hook.
    _locale_refresh_translations([static::LANGCODE], $deleted);
    _locale_refresh_configuration([static::LANGCODE], $deleted);

    $this->assertTextOnPage($path, $translationsToDelete, FALSE);
  }

  /**
   * Verifies whether the given list of strings is (not) present on the page.
   *
   * @param string $path
   *   The path which should be checked for the given strings.
   * @param list<string> $textToCheck
   *   The strings to check.
   * @param bool $mustBePresent
   *   Whether the text expected to be present or the opposite, it shouldn't
   *   appear.
   */
  protected function assertTextOnPage(
    string $path,
    array $textToCheck,
    bool $mustBePresent = TRUE,
  ): void {
    $this->drupalGet($path);
    $assertSession = $this->assertSession();
    $failingString = [];
    foreach ($textToCheck as $text) {
      if (
        ($mustBePresent && $assertSession->waitForText($text)) ||
        (!$mustBePresent && $this->waitForNoText($text))
      ) {
        continue;
      }
      $failingString[] = $text;
    }

    $this->assertEmpty(
      $failingString,
      sprintf(
        $mustBePresent
          ? "Missing string(s) which should be present on the page '%s':\n%s"
          : "String(s) which should not be present on the page '%s', but they are:\n%s",
        $path,
        implode("\n", $failingString)
      )
    );
  }

  /**
   * Asserts that text does not appear on page after a wait.
   *
   * Was copied from MediaLibraryTestBase; slightly modified to return boolean.
   *
   * @param string $text
   *   The text that should not be on the page.
   * @param int $timeout
   *   Timeout in milliseconds, defaults to 10000.
   *
   * @return bool
   *   Whether the given string cannot be found on the page.
   *
   * @todo replace with whatever gets added in
   *   https://www.drupal.org/node/3061852
   *
   * @see \Drupal\Tests\media_library\FunctionalJavascript\MediaLibraryTestBase::waitForNoText()
   */
  protected function waitForNoText(string $text, int $timeout = 10000): bool {
    $page = $this->getSession()->getPage();
    $result = $page->waitFor(
      $timeout / 1000,
      function (DocumentElement $page) use ($text): bool {
        $actual = preg_replace('/\s+/u', ' ', $page->getText());
        $regex = '/' . preg_quote($text, '/') . '/ui';
        return !preg_match($regex, $actual);
      }
    );
    return !empty($result);
  }

  /**
   * Removes base_url() and query args from file paths.
   *
   * Copied from JqueryUiLibraryAssetsTest.
   *
   * @param string $path
   *   The path being trimmed.
   *
   * @return string
   *   The trimmed path.
   *
   * @see \Drupal\FunctionalTests\Libraries\JqueryUiLibraryAssetsTest::trimFilePath()
   */
  protected function trimFilePath(string $path): string {
    $base_path_position = strpos($path, base_path());
    if ($base_path_position !== FALSE) {
      $path = substr_replace($path, '', $base_path_position, strlen(base_path()));
    }
    $query_pos = strpos($path, '?');
    return $query_pos !== FALSE ? substr($path, 0, $query_pos) : $path;
  }

}
