<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\BabelStringsRepositoryInterface;
use Drupal\babel\Model\Source;
use Drupal\Core\Asset\AssetResolverInterface;
use Drupal\Core\Asset\AttachedAssets;
use Drupal\Core\DestructableInterface;

// cspell:ignore spellout

/**
 * Tests how strings are added by the JavaScript alter hook.
 *
 * @see \babel_js_alter()
 */
class JsAlterTest extends BabelKernelTestBase {

  /**
   * Tests functionality of the JavaScript alter hook.
   *
   * @param list<int> $javaScriptLocaleLocations
   *   List of locale source IDs which are present in the locales_location
   *   table.
   * @param list<int> $localeSources
   *   List of locale source IDs which are present in the locales_source table.
   * @param list<int> $babelSources
   *   List of locale source IDs which are present
   *   in the babel_source_instance table.
   * @param list<int> $expectedBabelSources
   *   The list of the expected babel records after the JavaScript alter hook
   *   was invoked.
   *
   * @dataProvider providerJsAlterTest
   */
  public function testJavascriptAlter(
    array $javaScriptLocaleLocations,
    array $localeSources,
    array $babelSources,
    array $expectedBabelSources,
  ): void {
    $this->populateJsLocaleLocations($javaScriptLocaleLocations);
    $this->populateLocaleSources($localeSources);
    $this->populateBabelSources($babelSources);

    $this->triggerAlterHook();

    $stringsRepository = $this->container->get(BabelStringsRepositoryInterface::class);
    $babelStorage = $this->container->get(BabelStorageInterface::class);

    $localeStringIds = array_reduce(
      array_keys($stringsRepository->getStrings(static::LANGCODE)),
      function (array $ids, string $hash) use ($babelStorage): array {
        $localeIds = $babelStorage->getSourceStringInstances($hash)['locale'] ?? [];
        return array_unique([...$ids, ...$localeIds]);
      },
      [],
    );

    natsort($localeStringIds);
    $this->assertEquals($expectedBabelSources, array_values($localeStringIds));
  }

  /**
   * Data provider for ::testJavascriptAlter.
   *
   * @return array
   *   The test cases.
   */
  public static function providerJsAlterTest(): array {
    return [
      'No javascript strings present' => [
        'javaScriptLocaleLocations' => [],
        'localeSources' => [],
        'babelSources' => [],
        'expectedBabelSources' => [],
      ],
      'New strings should be recorded' => [
        'javaScriptLocaleLocations' => range(5, 6),
        'localeSources' => range(1, 6),
        'babelSources' => [],
        'expectedBabelSources' => range(5, 6),
      ],
      'No JS records' => [
        'javaScriptLocaleLocations' => [],
        'localeSources' => range(1, 6),
        'babelSources' => range(1, 6),
        'expectedBabelSources' => range(1, 6),
      ],
    ];
  }

  /**
   * Populates locale location table.
   *
   * @param list<int> $javaScriptLocaleLocations
   *   List of locale source IDs which should be present in the locales_location
   *   table.
   */
  protected function populateJsLocaleLocations(array $javaScriptLocaleLocations): void {
    if (!$javaScriptLocaleLocations) {
      return;
    }

    $records = array_map(
      function (int $identifier): array {
        return [
          'lid' => $identifier,
          'sid' => $identifier,
          'type' => 'javascript',
          'name' => 'foo',
          'version' => \Drupal::VERSION,
        ];
      },
      $javaScriptLocaleLocations
    );

    $query = \Drupal::database()->insert('locales_location')
      ->fields(array_keys(current($records)));
    foreach ($records as $record) {
      $query->values($record);
    }
    $query->execute();
  }

  /**
   * Populates locale source table.
   *
   * @param list<int> $localeSources
   *   List of locale source IDs which should be present in the table.
   */
  protected function populateLocaleSources(array $localeSources): void {
    if (!$localeSources) {
      return;
    }

    $records = array_map(
      function (int $identifier): array {
        return [
          'lid' => $identifier,
          // If this fails then the test environment misses ext-intl.
          'source' => \NumberFormatter::create('en', \NumberFormatter::SPELLOUT)->format($identifier),
          'context' => '',
          'version' => \Drupal::VERSION,
        ];
      },
      $localeSources
    );

    $query = \Drupal::database()->insert('locales_source')
      ->fields(array_keys(current($records)));
    foreach ($records as $record) {
      $query->values($record);
    }
    $query->execute();
  }

  /**
   * Populates the babel source table.
   *
   * @param list<int> $babelSources
   *   List of locale source IDs which should be present
   *   in the babel_source_instance table.
   */
  protected function populateBabelSources(array $babelSources): void {
    if (!$babelSources) {
      return;
    }

    $records = array_map(
      function (int $identifier): array {
        // If this fails then the test environment misses ext-intl.
        $string = \NumberFormatter::create('en', \NumberFormatter::SPELLOUT)->format($identifier);
        $source = new Source($string, '');
        return [
          'plugin' => 'locale',
          'id' => $identifier,
          'hash' => $source->getHash(),
          'sort_key' => $source->getSortKey(),
        ];
      },
      $babelSources
    );

    $db = \Drupal::database();
    $querySource = $db->insert('babel_source_instance')
      ->fields(['plugin', 'id', 'hash']);
    $queryStatus = $db->insert('babel_source')
      ->fields(['hash', 'status', 'sort_key']);
    foreach ($records as $record) {
      $querySource->values([
        'plugin' => $record['plugin'],
        'id' => $record['id'],
        'hash' => $record['hash'],
      ]);
      $queryStatus->values([
        'hash' => $record['hash'],
        'status' => 1,
        'sort_key' => $record['sort_key'],
      ]);
    }
    $querySource->execute();
    $queryStatus->execute();
  }

  /**
   * Triggers JS alter hooks being invoked.
   */
  protected function triggerAlterHook(): void {
    $testAssets = (new AttachedAssets())
      ->setAlreadyLoadedLibraries([])
      ->setLibraries(['core/drupal'])
      ->setSettings(['currentTime' => time()]);

    $assetResolver = \Drupal::service('asset.resolver');
    $this->assertInstanceOf(AssetResolverInterface::class, $assetResolver);

    // Trigger alter hook.
    $assetResolver->getJsAssets($testAssets, FALSE, $this->language);

    // Invoke destruct on our services depending on implementation.
    foreach (['string_translator.locale.lookup', BabelStorageInterface::class] as $id) {
      $service = \Drupal::service($id);
      if ($service instanceof DestructableInterface) {
        $service->destruct();
      }
    }
  }

}
