<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel\Plugin\Babel\DataTransfer;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Model\Translation;
use Drupal\babel\Plugin\Babel\DataTransferPluginInterface;
use Drupal\babel\Plugin\Babel\DataTransferPluginManager;
use Drupal\Component\Gettext\PoItem;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;

/**
 * Tests the Spreadsheet transfer plugin.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel\Plugin\Babel\DataTransfer\Spreadsheet
 */
class SpreadsheetTest extends BabelKernelTestBase {

  /**
   * The plugin instance to be used in the test.
   */
  protected ?DataTransferPluginInterface $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->plugin = $this->container->get(DataTransferPluginManager::class)
      ->createInstance('spreadsheet');
  }

  /**
   * @covers ::createExportedFileContent
   *
   * @dataProvider providerTestExport
   */
  public function testCreateExportFile(
    StringTranslation $stringTranslation,
    string $stringTranslationId,
    string $expected,
  ): void {
    $exportedCsv = $this->plugin->createExportedFileContent([$stringTranslationId => $stringTranslation], 'fi', 'csv');
    $this->assertStringContainsString($expected, $exportedCsv);
  }

  /**
   * Test data provider for ::testCreateExportFile.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestExport(): array {
    return [
      'String without translation' => [
        'stringTranslation' => new StringTranslation(
          new Source('Translation', '', TRUE),
          NULL,
        ),
        'stringTranslationId' => 'locale:1',
        'expected' => <<<EOS
          "Translation","","","","","locale:1"
          EOS,
      ],
      'String with translation' => [
        'stringTranslation' => new StringTranslation(
          new Source('Translation', '', TRUE),
          new Translation('fi', '[FI] Translation'),
        ),
        'stringTranslationId' => 'locale:2',
        'expected' => <<<EOS
          "Translation","[FI] Translation","","","","locale:2"
          EOS,
      ],
      'Plural' => [
        'stringTranslation' => new StringTranslation(
          new Source('One item' . PoItem::DELIMITER . '@count items', 'num of items', TRUE),
          new Translation('fi', '[FI] One item' . PoItem::DELIMITER . '[FI] @count items'),
        ),
        'stringTranslationId' => 'locale:3',
        'expected' => <<<EOS
          "One item","[FI] One item","num of items","singular","","locale:3"
          "@count items","[FI] @count items","","plural","","locale:3"
          EOS,
      ],
      'Disabled' => [
        'stringTranslation' => new StringTranslation(
          new Source('Baz', '', FALSE),
          new Translation('fi', '[FI] Baz'),
        ),
        'stringTranslationId' => 'locale:4',
        'expected' => '',
      ],
    ];
  }

  /**
   * Tests how a CSV gets imported.
   *
   * @param string $csvContent
   *   Content of the file to import.
   * @param array $expectedStrings
   *   The expected translations.
   *
   * @covers ::getImportedTranslations
   *
   * @dataProvider providerTestImport
   */
  public function testGetImportedTranslations(
    string $csvContent,
    array $expectedStrings,
  ): void {
    $this->container->get(BabelStorageInterface::class)->update('locale', [
      new Source('Foo', ''),
      new Source("One bar\03@count bars:", ''),
      new Source('Untranslated baz', 'location of sth'),
    ]);
    $tempFilePath = $this->createTempFile($csvContent);
    $this->assertEquals($expectedStrings, $this->plugin->getImportedTranslations($tempFilePath, 'es'));
  }

  /**
   * Test data provider for ::testGetImportedTranslations.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestImport(): array {
    $hash = [
      'Foo:' => (new Source('Foo', ''))->getHash(),
      "One bar\03@count bars:" => (new Source("One bar\03@count bars:", ''))->getHash(),
      'Untranslated baz:location of sth' => (new Source('Untranslated baz', 'location of sth'))->getHash(),
    ];
    return [
      'Import' => [
        'csvContent' => <<<EOS
          "Source string","Translated string","Context","Notes","URL","ID (do not edit)"
          "Foo","[ES] Foo","location of sth","","","{$hash['Foo:']}"
          "One bar","[ES] One bar","location of sth","","","{$hash["One bar\03@count bars:"]}"
          "@count bars","[ES] @count bars","location of sth","","","{$hash["One bar\03@count bars:"]}"
          "Untranslated baz","","location of sth","","","{$hash['Untranslated baz:location of sth']}"
          "Corrupted row"
          EOS,
        'expectedStrings' => [
          $hash['Foo:'] => [
            '[ES] Foo',
          ],
          $hash["One bar\03@count bars:"] => [
            '[ES] One bar',
            '[ES] @count bars',
          ],
        ],
      ],
    ];
  }

  /**
   * Tests import errors.
   *
   * @param string $csvContent
   *   The content of the CSV import to check for errors.
   * @param string[] $expectedErrors
   *   The expected errors.
   *
   * @covers ::getImportErrors
   *
   * @dataProvider providerTestErrors
   */
  public function testGetImportErrors(
    string $csvContent,
    array $expectedErrors,
  ): void {
    $this->plugin->getImportedTranslations($this->createTempFile($csvContent), 'ro');

    $this->assertEquals(
      $expectedErrors,
      array_map(
        fn (string|\Stringable $msg) => (string) $msg,
        $this->plugin->getImportErrors(),
      ),
    );
  }

  /**
   * Test data provider for ::testGetImportErrors.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestErrors(): array {
    return [
      'Invalid headers' => [
        'csvContent' => <<<EOS
          "Source string",TRANSLATED,Context,"Notes","URL"
          "Duplicated","ID","","","","config:a"
          "is","OK","","","","config:a"
          EOS,
        'expectedErrors' => [
          'Invalid header at B1: expected <em class="placeholder">Translated string</em>, got <em class="placeholder">&#039;TRANSLATED&#039;</em>.',
          'Invalid header at F1: expected <em class="placeholder">ID (do not edit)</em>, got <em class="placeholder">NULL</em>.',
        ],
      ],
    ];
  }

  /**
   * Tests import warnings.
   *
   * @param string $csvContent
   *   The content of the CSV import to check for warnings.
   * @param string[] $expectedWarnings
   *   The expected warnings.
   *
   * @covers ::getImportWarnings
   *
   * @dataProvider providerTestWarnings
   */
  public function testGetImportWarnings(
    string $csvContent,
    array $expectedWarnings,
  ): void {
    $this->container->get(BabelStorageInterface::class)->update('locale', [
      new Source('Another', ''),
      new Source('duplicated', ''),
      new Source('Foo', ''),
    ]);

    // Trigger import.
    $this->plugin->getImportedTranslations($this->createTempFile($csvContent), 'ro');

    $this->assertEquals(
      $expectedWarnings,
      array_map(
        fn (string|\Stringable $msg) => (string) $msg,
        $this->plugin->getImportWarnings(),
      ),
    );
  }

  /**
   * Test data provider for ::testGetImportWarnings.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestWarnings(): array {
    $hash = [
      'Another:' => (new Source('Another', ''))->getHash(),
      'duplicated:' => (new Source('duplicated', ''))->getHash(),
      'Foo:' => (new Source('Foo', ''))->getHash(),
    ];

    return [
      'Invalid import' => [
        'csvContent' => <<<EOS
          "Source string","Translated string","Context","Notes","URL","ID (do not edit)"
          "Translation","[FI] Translation","","",""
          "Missing plugin","","","","","1111"
          "Another","attempt","","","","{$hash['Another:']}"
          "duplicated","id","","","","{$hash['duplicated:']}"
          EOS,
        'expectedWarnings' => [
          'The translation identifier at cell F2 is empty. Did you edit that column?',
          "The translation identifier at cell F3 is invalid. It should be a 64 characters alphanumeric string. Did you edit that column?",
        ],
      ],
      'Perfect CSV' => [
        'csvContent' => <<<EOS
          "Source string","Translated string","Context","Notes","URL","ID (do not edit)"
          "Foo","Bar","","","","{$hash['Foo:']}"
          EOS,
        'expectedWarnings' => [],
      ],
    ];
  }

  /**
   * Puts the given string into a temporary file and returns the file location.
   *
   * @param string $fileContent
   *   The content of the file.
   *
   * @return string
   *   The path to the temporary file.
   */
  protected function createTempFile(string $fileContent): string {
    $tmpfilePath = stream_get_meta_data(tmpfile())['uri'];
    file_put_contents($tmpfilePath, $fileContent);
    return $tmpfilePath;
  }

}
