<?php

namespace Drupal\babel\Form;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\BabelStringsRepositoryInterface;
use Drupal\babel\Plugin\Babel\DataTransferPluginManager;
use Drupal\Component\Utility\Html;
use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\file\FileInterface;

/**
 * Export form.
 */
class BabelExportForm extends FormBase {

  use AjaxHelperTrait;
  use AutowireTrait;

  public function __construct(
    protected DataTransferPluginManager $dataTransferManager,
    protected BabelStorageInterface $babelStorage,
    protected LanguageManagerInterface $languageManager,
    protected FileSystemInterface $fileSystem,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected BabelStringsRepositoryInterface $stringsRepository,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $options = [];
    foreach ($this->dataTransferManager->getExporters() as $pluginId => $definition) {
      $options[$pluginId] = $definition['label'];
    }

    $languages = array_map(
      fn(LanguageInterface $language): string => $language->getName(),
      $this->languageManager->getLanguages(),
    );
    unset($languages['en']);

    $form['language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $languages,
      '#required' => TRUE,
    ];
    $form['plugin'] = [
      '#type' => 'select',
      '#title' => $this->t('Exporter'),
      '#options' => $options,
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::refreshExtensions',
      ],
    ];

    $pluginId = $form_state->getValue('plugin') ?? $form_state->getUserInput()['plugin'] ?? NULL;

    if ($pluginId) {
      $definition = $this->dataTransferManager->getDefinition($pluginId);

      $form['extension'] = [
        '#type' => 'select',
        '#title' => $this->t('Extension'),
        '#options' => array_map(
          fn(string $extension): string => ".$extension",
          array_combine($definition['fileExtensions'], $definition['fileExtensions']),
        ),
        '#required' => TRUE,
      ];
    }
    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Export'),
      ],
    ];

    if ($this->isAjax()) {
      // @see https://www.drupal.org/node/2897377
      $form['#id'] = Html::getId($form_state->getBuildInfo()['form_id']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $langcode = $form_state->getValue('language');
    $extension = $form_state->getValue('extension');
    $strings = $this->stringsRepository->getStrings($langcode);

    $plugin = $this->dataTransferManager->createInstance($form_state->getValue('plugin'));
    $content = $plugin->createExportedFileContent($strings, $langcode, $extension);

    $dataTransfer = $this->config('babel.settings')->get('data_transfer');
    $destination = $dataTransfer['destination'];
    if ($prefix = $export['prefix'] ?? '') {
      $prefix .= '-';
    }

    $this->fileSystem->prepareDirectory($destination, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    $filePath = $destination . "/$prefix$langcode.$extension";
    file_put_contents($filePath, $content);

    $storage = $this->entityTypeManager->getStorage('file');
    $file = $storage->create(['uri' => $filePath]);
    assert($file instanceof FileInterface);
    $file->setTemporary();
    $file->save();

    $this->messenger()->addStatus($this->t('The %lang translation file has been exported. Download the @type file <a href=":url">here</a>.', [
      '%lang' => $this->languageManager->getLanguage($langcode)->getName(),
      '@type' => $plugin->getPluginDefinition()['label'],
      ':url' => $file->createFileUrl(),
    ]));
  }

  /**
   * Provided an Ajax submit callback.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state instance.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The Ajax response.
   */
  public function refreshExtensions(array $form, FormStateInterface $formState): AjaxResponse {
    $selector = '[data-drupal-selector="' . $form['#attributes']['data-drupal-selector'] . '"]';
    return (new AjaxResponse())->addCommand(new ReplaceCommand($selector, $form));
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'babel_export_form';
  }

}
