<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel\Plugin\Babel\TranslationType;

use Drupal\babel\Model\Source;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Tests\babel\Traits\ModuleInstallHelperTrait;

/**
 * Tests the "config" translation type plugin.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel\Plugin\Babel\TranslationType\Config
 */
class ConfigTest extends TranslationTypePluginTestBase {

  use ModuleInstallHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->runInstallBatchOperations();
    $this->plugin = $this->container
      ->get(TranslationTypePluginManager::class)
      ->createInstance('config');
  }

  /**
   * {@inheritdoc}
   *
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringIds
   *   The string IDs to load.
   * @param array $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetMultiple
   */
  public function testGetStrings(
    string $langcode = '',
    array $stringIds = [],
    array $expected = [],
  ): void {
    $this->assertEquals(
      $expected,
      array_keys($this->plugin->getStrings($langcode, $stringIds))
    );
  }

  /**
   * Test data provider for ::testGetStrings.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetMultiple(): array {
    return [
      'Load everything' => [
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => [
          'language.entity.en:label',
          'language.entity.pt:label',
          'language.entity.und:label',
          'language.entity.zxx:label',
        ],
      ],
      'Load specific' => [
        'langcode' => 'fr',
        'stringIds' => ['language.entity.und:label'],
        'expected' => [
          'language.entity.und:label',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function testGetString(): void {
    $this->assertEquals(
      [
        'source' => new Source(
          string: 'Not applicable',
          context: '',
          status: TRUE,
        ),
        'translation' => NULL,
      ],
      $this->plugin->getString('pl', 'language.entity.zxx:label')->toArray()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function testUpdateTranslation(): void {
    $languageManager = \Drupal::languageManager();
    $original = $languageManager->getConfigOverrideLanguage();
    $configFactory = \Drupal::configFactory();
    $configLanguageStorage = \Drupal::entityTypeManager()->getStorage('configurable_language');
    $this->assertInstanceOf(ConfigEntityStorageInterface::class, $configLanguageStorage);

    // Check object before translation was added.
    $languageManager->setConfigOverrideLanguage($this->language);
    // First, check as config object.
    $this->assertEquals('Not applicable', $configFactory->get('language.entity.zxx')->get('label'));
    // Next, check as config entity.
    $nonApplicableLanguageEntity = $configLanguageStorage->load('zxx');
    $this->assertEquals('Not applicable', $nonApplicableLanguageEntity->label());
    $languageManager->setConfigOverrideLanguage($original);

    // Add translation.
    $this->plugin->updateTranslation(
      $this->plugin->getString(
        $this->language->getId(),
        'language.entity.zxx:label'
      ),
      'language.entity.zxx:label',
      $this->language->getId(),
      '[TRANSLATED] Not applicable',
    );

    // Check the result.
    $languageManager->setConfigOverrideLanguage($this->language);
    // Check as config object.
    $this->assertEquals('[TRANSLATED] Not applicable', $configFactory->get('language.entity.zxx')->get('label'));
    // Check as config entity.
    $nonApplicableLanguageEntityUpdated = $configLanguageStorage->load('zxx');
    $this->assertEquals('[TRANSLATED] Not applicable', $nonApplicableLanguageEntityUpdated->label());
  }

}
