<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_menu_link_content\Kernel;

use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;
use Drupal\Tests\babel\Traits\ModuleInstallHelperTrait;

/**
 * Tests operations of BabelMenuLinkContentService on install.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel_menu_link_content\BabelMenuLinkContentService
 */
class BabelMenuLinkContentServiceInstallTest extends BabelKernelTestBase {

  use ModuleInstallHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['menu_link_content', 'link'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('menu_link_content');
  }

  /**
   * Tests how menu link content entities are discovered after installing Babel.
   *
   * @param array $menuLinkValues
   *   The menu link values.
   * @param list<string> $expectedProperties
   *   The expected discovered translatable menu link properties.
   *
   * @covers ::batchAddSources
   *
   * @dataProvider providerTestInstall
   */
  public function testInstall(
    array $menuLinkValues,
    array $expectedProperties,
  ): void {
    $storage = \Drupal::entityTypeManager()->getStorage('menu_link_content');
    $sampleMenuLink = $storage->create($menuLinkValues);
    $this->assertEquals(SAVED_NEW, $sampleMenuLink->save());

    // Install the module.
    $this->container->get('module_installer')->install(['babel_menu_link_content']);
    $this->runInstallBatchOperations();

    $pluginManager = \Drupal::service(TranslationTypePluginManager::class);
    $this->assertInstanceOf(TranslationTypePluginManager::class, $pluginManager);
    $plugin = $pluginManager->createInstance('menu_link_content');

    $this->assertEquals($expectedProperties, array_keys($plugin->getStrings('fr')));
  }

  /**
   * Test data for ::testInstall.
   *
   * @return array[]
   *   Test cases.
   */
  public static function providerTestInstall(): array {
    return [
      'Most menu link properties' => [
        'menuLinkValues' => [
          'menu_name' => 'test',
          'title' => 'Test menu link',
          'link' => [
            'title' => 'Link title',
            'uri' => 'https://example.com/',
          ],
          'description' => 'Some description',
          'langcode' => 'en',
        ],
        'expectedProperties' => [
          'test:1:description:0',
          'test:1:title:0',
        ],
      ],

      'Only title' => [
        'menuLinkValues' => [
          'menu_name' => 'test',
          'title' => 'No description',
          'link' => [
            'title' => 'Link title',
            'uri' => 'https://example.com/',
          ],
          'langcode' => 'en',
        ],
        'expectedProperties' => [
          'test:1:title:0',
        ],
      ],

      'Not applicable langcode' => [
        'menuLinkValues' => [
          'menu_name' => 'test',
          'title' => '🤷‍♂️',
          'link' => [
            'title' => 'Link title',
            'uri' => 'https://example.com/',
          ],
          'langcode' => LanguageInterface::LANGCODE_NOT_APPLICABLE,
        ],
        'expectedProperties' => [
          'test:1:title:0',
        ],
      ],
    ];
  }

}
