<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Drupal\Core\Url;

/**
 * Tests translation activation and deactivation functionality.
 *
 * @group babel
 */
class BabelSourceStatusTest extends BabelFunctionalJavascriptTestBase {

  /**
   * Test strings for translation activation/deactivation.
   */
  protected const TEST_STRINGS = [
    'Administrative task links',
    'An AJAX HTTP error occurred.',
  ];

  /**
   * Test disable and enable for specific strings.
   */
  public function testStringActivationDeactivation(): void {
    $this->drupalLogin($this->testAccount);
    $this->drupalGet(Url::fromRoute('babel.ui', ['language' => static::LANGCODE]));

    foreach (static::TEST_STRINGS as $testString) {
      $this->assertStringToggle($testString);
    }
  }

  /**
   * Test disable/enable toggle for a specific string.
   */
  protected function assertStringToggle(string $testString): void {
    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();
    $page->hasContent($testString);

    $xpath = "//tr[.//text()[contains(., '$testString')]]//td[contains(@class, 'babel--activate')]//input";
    $button = $page->find('xpath', $xpath);
    $this->assertNotNull($button, "Button not found for: $testString");

    $active = $button->hasClass('activated');
    $activeTitle = $button->getAttribute('title');

    $button->click();
    $this->assertSession()->assertWaitOnAjaxRequest();

    $page->findField('include_inactive')->check();
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();

    $button = $page->find('xpath', $xpath);
    $this->assertNotNull($button, "Button should still exist after click");

    $inactive = $button->hasClass('activated');
    $inactiveTitle = $button->getAttribute('title');

    $this->assertNotEquals($active, $inactive, "State should change for: $testString");
    $this->assertNotEquals($activeTitle, $inactiveTitle, "Title should change for: $testString");

    $button->click();
    $this->assertSession()->assertWaitOnAjaxRequest();

    $page->findField('translated')->selectOption('untranslated');

    $newActive = $button->hasClass('activated');
    $newActiveTitle = $button->getAttribute('title');

    $this->assertEquals($active, $newActive, "State should change for: $testString");
    $this->assertEquals($activeTitle, $newActiveTitle, "Title should change for: $testString");
  }

}
