<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel\EventSubscriber;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\EventSubscriber\ConfigSubscriber;
use Drupal\babel\Model\Source;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigRenameEvent;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;

/**
 * Tests the configuration subscriber's rename cleanup functionality.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel\EventSubscriber\ConfigSubscriber
 */
class ConfigSubscriberRenameTest extends BabelKernelTestBase {

  /**
   * The config subscriber under test.
   *
   * @var \Drupal\babel\EventSubscriber\ConfigSubscriber
   */
  protected ConfigSubscriber $subscriber;

  /**
   * The Babel storage service.
   *
   * @var \Drupal\babel\BabelStorageInterface
   */
  protected BabelStorageInterface $babelStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->subscriber = $this->container->get(ConfigSubscriber::class);
    $this->babelStorage = $this->container->get(BabelStorageInterface::class);
  }

  /**
   * Tests that renamed configuration entries are properly cleaned up.
   *
   * @covers ::onConfigRename
   */
  public function testConfigRenameCleanup(): void {
    $oldConfigName = 'block.block.test_block-old_name';
    $newConfigName = 'block.block.test_block_old_name';

    $testEntries = [
      "$oldConfigName:settings.label" => 'Test Block Label',
      "$oldConfigName:settings.description" => 'Test Block Description',
    ];

    foreach ($testEntries as $id => $value) {
      $this->babelStorage->update('config', [$id => new Source($value, '')]);
    }

    $entriesBeforeRename = $this->getEntriesForConfig($oldConfigName);
    $this->assertCount(2, $entriesBeforeRename, 'Old entries should exist before rename.');

    $mockConfig = $this->createMock(Config::class);
    $mockConfig->method('getName')->willReturn($newConfigName);
    $renameEvent = new ConfigRenameEvent($mockConfig, $oldConfigName);

    $this->subscriber->onConfigRename($renameEvent);

    $entriesAfterRename = $this->getEntriesForConfig($oldConfigName);
    $this->assertCount(0, $entriesAfterRename, 'Old Babel entries should be deleted after config rename.');
  }

  /**
   * Helper method to get Babel entries for a specific configuration.
   *
   * @param string $configName
   *   The configuration name to search for.
   *
   * @return array
   *   Array of Babel entry IDs for the given configuration.
   */
  private function getEntriesForConfig(string $configName): array {
    $allEntries = $this->babelStorage->getBaseQuery('config')
      ->execute()
      ->fetchAllKeyed(0, 0);

    $matchingEntries = [];
    $prefix = "$configName:";
    foreach ($allEntries as $id => $value) {
      if (str_starts_with($id, $prefix)) {
        $matchingEntries[$id] = $value;
      }
    }

    return $matchingEntries;
  }

}
