<?php

declare(strict_types=1);

namespace Drupal\babel_menu_link_content;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\menu_link_content\MenuLinkContentInterface;

/**
 * Installation helper.
 */
class BabelMenuLinkContentService {

  /**
   * List of the menu link entity properties which aren't translated.
   *
   * @const string[]
   */
  private const EXCLUDED_FIELDS = [
    'content_translation_created',
    'content_translation_outdated',
    'content_translation_source',
    'content_translation_status',
    'content_translation_uid',
    'default_langcode',
    'langcode',
    'changed',
    'revision_translation_affected',
  ];

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly LanguageManagerInterface $languageManager,
  ) {}

  /**
   * Updates the source objects of the given menu link entity.
   *
   * @param \Drupal\menu_link_content\MenuLinkContentInterface $link
   *   The menu link content entity to process.
   */
  public function updateSourcesForLink(MenuLinkContentInterface $link): void {
    // Avoid duplicate sources created from translations of menu links.
    if ($link->language()->getId() !== $this->languageManager->getDefaultLanguage()->getId()) {
      return;
    }
    // Cleanup first.
    $this->babelStorage->delete('menu_link_content', $link->getMenuName() . ':' . $link->id() . ':');
    $this->babelStorage->update('menu_link_content', $this->getSourcesForLink($link));
  }

  /**
   * Instantiates source objects from the properties of the given menu link.
   *
   * @param \Drupal\menu_link_content\MenuLinkContentInterface $link
   *   The menu link content entity to process.
   *
   * @return list<\Drupal\babel\Model\Source>
   *   List of source objects.
   */
  public function getSourcesForLink(MenuLinkContentInterface $link): array {
    $sources = [];

    foreach ($link->getTranslatableFields(FALSE) as $field) {
      if (in_array($field->getName(), self::EXCLUDED_FIELDS, TRUE)) {
        continue;
      }

      foreach ($field as $delta => $item) {
        $id = $link->getMenuName() . ':' . $link->id() . ':' . $field->getName() . ':' . $delta;
        $sources[$id] = new Source(string: (string) $item->value, context: '');
      }
    }

    return $sources;
  }

  /**
   * Creates a batch process to process preexisting menu links entities.
   *
   * @param array $menuIds
   *   ID of the menus which menu link entities must be processed.
   */
  public function batchAddSources(array $menuIds = []): void {
    $menuLinkIdQuery = $this->entityTypeManager->getStorage('menu_link_content')
      ->getQuery()
      ->accessCheck(FALSE);
    if ($menuIds) {
      $menuLinkIdQuery->condition('menu_name', $menuIds, 'IN');
    }
    $menuLinkIds = $menuLinkIdQuery->execute();
    if (!$menuLinkIds) {
      return;
    }
    $total = count($menuLinkIds);

    $batch = new BatchBuilder();
    foreach (array_chunk($menuLinkIds, 10) as $chunk) {
      $batch->addOperation([BabelMenuLinkContentService::class, 'processMenuLinks'], [$chunk, $total]);
    }

    batch_set($batch->toArray());
  }

  /**
   * Batch callback to process menu link content entities.
   *
   * @param array<string|int> $lids
   *   ID of menu link content entities.
   * @param int $total
   *   Total number of items to process. Only used to construct the interactive
   *   message.
   * @param array $context
   *   The batch context.
   */
  public static function processMenuLinks(array $lids, int $total, array &$context): void {
    $default_language = \Drupal::languageManager()->getDefaultLanguage();
    $context['results']['progress'] ??= 0;

    $storage = \Drupal::entityTypeManager()->getStorage('menu_link_content');
    $service = \Drupal::service(BabelMenuLinkContentService::class);

    $sources = [];
    foreach ($storage->loadMultiple($lids) as $link) {
      // Avoid duplicate sources created from translations of menu links.
      if ($link->language()->getId() !== $default_language->getId()) {
        continue;
      }
      $sources = [...$sources, ...$service->getSourcesForLink($link)];
    }
    \Drupal::service(BabelStorageInterface::class)->update('menu_link_content', $sources);

    $context['results']['progress'] += count($lids);
    $context['message'] = t('Processed @progress out of @total menu links', [
      '@progress' => $context['results']['progress'],
      '@total' => $total,
    ]);
  }

}
