<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_menu_link_content\Kernel;

use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\babel_menu_link_content\Plugin\Babel\TranslationType\MenuLinkContent;
use Drupal\Component\Utility\NestedArray;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\Tests\babel\Kernel\BabelKernelTestBase;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * Tests the BabelMenuLinkContentService service.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel_menu_link_content\BabelMenuLinkContentService
 */
class BabelMenuLinkContentServiceTest extends BabelKernelTestBase {

  use ProphecyTrait;

  /**
   * Static cache for the menu link content entity created in testCreate.
   *
   * @var \Drupal\menu_link_content\MenuLinkContentInterface
   */
  protected MenuLinkContentInterface $menuLinkContent;

  /**
   * A "menu_link_content" translation type plugin instance.
   */
  protected MenuLinkContent $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->container->get('module_installer')->install(['babel_menu_link_content']);
    $this->plugin = $this->container
      ->get(TranslationTypePluginManager::class)
      ->createInstance('menu_link_content');
  }

  /**
   * Tests whether newly created menu links are properly discovered.
   */
  public function testCreate(): void {
    $storage = \Drupal::entityTypeManager()->getStorage('menu_link_content');

    $this->menuLinkContent = $storage->create($this->mergeRequiredLinkValues([
      'title' => 'Test menu link',
      'description' => 'Some description',
      'link' => ['uri' => 'http://example.com/test'],
    ]));
    $this->menuLinkContent->save();

    // Menu links not in sites default language should not generate sources.
    $storage->create($this->mergeRequiredLinkValues([
      'title' => static::LANGCODE . 'menu link',
      'description' => 'description',
      'link' => ['uri' => 'http://example.com/' . static::LANGCODE . 'test/'],
      'langcode' => static::LANGCODE,
    ]))->save();

    $this->assertTrue(TRUE);
    $this->assertEquals(
      [
        'test:1:description:0' => new StringTranslation(
          source: new Source('Some description', '', TRUE),
        ),
        'test:1:title:0' => new StringTranslation(
          source: new Source('Test menu link', '', TRUE),
        ),
      ],
      $this->plugin->getStrings('fr'),
    );
  }

  /**
   * Tests whether changes of menu links are properly recorder.
   *
   * @depends testCreate
   */
  public function testUpdate(): void {
    $this->testCreate();

    // Translations should not register sources to avoid duplicates.
    $this->menuLinkContent->addTranslation(static::LANGCODE, $this->menuLinkContent->toArray());

    $this->menuLinkContent
      ->set('title', 'New title')
      ->set('description', NULL)
      ->save();

    $this->assertEquals(
      [
        'test:1:title:0' => new StringTranslation(
          source: new Source('New title', '', TRUE),
        ),
      ],
      $this->plugin->getStrings('cz'),
    );
  }

  /**
   * Tests cleanup when menu links are deleted.
   *
   * @depends testCreate
   */
  public function testDelete(): void {
    $this->testCreate();

    $this->menuLinkContent->delete();

    $this->assertEquals([], $this->plugin->getStrings('ro'));
  }

  /**
   * Merges some mandatory menu link properties.
   *
   * @param array $values
   *   Menu link data to merge the required values into.
   */
  protected function mergeRequiredLinkValues(array $values): array {
    $defaults = [
      'langcode' => 'en',
      'menu_name' => 'test',
    ];
    $defaults['link'] = !array_key_exists('link', $values)
      ? 'https://example.com/'
      : ['uri' => 'https://example.com/'];
    return NestedArray::mergeDeepArray([$defaults, $values], TRUE);
  }

}
