<?php

declare(strict_types=1);

namespace Drupal\Tests\babel_menu_link_content\Kernel\Plugin\Babel\TranslationType;

use Drupal\babel\Model\Source;
use Drupal\babel\Model\StringTranslation;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\system\Entity\Menu;
use Drupal\Tests\babel\Kernel\Plugin\Babel\TranslationType\TranslationTypePluginTestBase;

/**
 * Tests the MenuLinkContent translation type plugin.
 *
 * @group babel
 *
 * @coversDefaultClass \Drupal\babel_menu_link_content\Plugin\Babel\TranslationType\MenuLinkContent
 */
class MenuLinkContentTest extends TranslationTypePluginTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'link',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->container->get('module_installer')->install(['babel_menu_link_content']);
    $this->plugin = $this->container
      ->get(TranslationTypePluginManager::class)
      ->createInstance('menu_link_content');

    Menu::create([
      'id' => 'test',
      'label' => 'Test menu',
      'description' => 'Description text',
    ])->save();

    \Drupal::service('content_translation.manager')->setEnabled('menu_link_content', 'menu_link_content', TRUE);
  }

  /**
   * {@inheritdoc}
   *
   * @param int $numberOfPreexistingLinks
   *   Number of the preexisting links.
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringIds
   *   The string IDs to load.
   * @param array $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetMultiple
   */
  public function testGetStrings(
    int $numberOfPreexistingLinks = 0,
    string $langcode = '',
    array $stringIds = [],
    array $expected = [],
  ): void {
    if ($numberOfPreexistingLinks > 0) {
      while ($numberOfPreexistingLinks > 0) {
        $this->createRandomMenuLink();
        $numberOfPreexistingLinks--;
      }
    }
    $this->assertEquals(
      $expected,
      array_keys($this->plugin->getStrings($langcode, $stringIds))
    );
  }

  /**
   * Test data provider for ::testGetStrings.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetMultiple(): array {
    return [
      'Load everything, no link' => [
        'numberOfPreexistingLinks' => 0,
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => [],
      ],
      'Load everything, some links' => [
        'numberOfPreexistingLinks' => 1,
        'langcode' => 'fi',
        'stringIds' => [],
        'expected' => ['test:1:title:0'],
      ],
      'Load specific' => [
        'numberOfPreexistingLinks' => 3,
        'langcode' => 'be',
        'stringIds' => ['test:2:title:0'],
        'expected' => [
          'test:2:title:0',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param int $numberOfPreexistingLinks
   *   Number of the preexisting links.
   * @param string $langcode
   *   The language code to test the loading with.
   * @param array $stringId
   *   The string ID to load.
   * @param \Drupal\babel\Model\StringTranslation|null $expected
   *   The expected result.
   *
   * @dataProvider providerTestGetSingle
   */
  public function testGetString(
    int $numberOfPreexistingLinks = 0,
    string $langcode = '',
    string $stringId = '',
    ?StringTranslation $expected = NULL,
  ): void {
    if ($numberOfPreexistingLinks > 0) {
      while ($numberOfPreexistingLinks > 0) {
        $this->createRandomMenuLink();
        $numberOfPreexistingLinks--;
      }
    }

    $this->assertEquals(
      $expected,
      $this->plugin->getString($langcode, $stringId)
    );
  }

  /**
   * Test data provider for ::testGetString.
   *
   * @return array[]
   *   The test cases.
   */
  public static function providerTestGetSingle(): array {
    return [
      'Load specific ID' => [
        'numberOfPreexistingLinks' => 2,
        'langcode' => 'fi',
        'stringId' => 'test:2:title:0',
        'expected' => new StringTranslation(
          source: new Source('Link 2 title', '', TRUE),
        ),
      ],
      'Missing ID' => [
        'numberOfPreexistingLinks' => 0,
        'langcode' => 'fi',
        'stringId' => 'test:2:title:0',
        'expected' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function testUpdateTranslation(): void {
    $link = $this->createRandomMenuLink('Test link title', 'Description');

    $this->assertTrue($link->isTranslatable());
    $this->assertFalse($link->hasTranslation($this->language->getId()));

    // Add translation.
    $this->plugin->updateTranslation(
      $this->plugin->getString(
        $this->language->getId(),
        'test:1:title:0',
      ),
      'test:1:title:0',
      $this->language->getId(),
      '[TRANSLATED] Test link title',
    );

    $link = \Drupal::entityTypeManager()->getStorage('menu_link_content')->load($link->id());

    $this->assertTrue($link->hasTranslation($this->language->getId()));
    $this->assertEquals('[TRANSLATED] Test link title', $link->getTranslation($this->language->getId())->getTitle());
  }

  /**
   * Creates a random menu link.
   *
   * @param string|null $title
   *   The title to use. Optional, by default this follows the next pattern:
   *   "Link <link-id> title".
   * @param string|null $description
   *   The menu link description. Optional.
   */
  protected function createRandomMenuLink(?string $title = NULL, ?string $description = NULL): MenuLinkContentInterface {
    $storage = \Drupal::entityTypeManager()->getStorage('menu_link_content');
    $nextId = 1;
    $highestIdResult = $storage->getQuery()->accessCheck(FALSE)->sort('id', 'DESC')->range(0, 1)->execute();
    $nextId += current($highestIdResult) ?: 0;

    $link = $storage->create(array_filter([
      'menu_name' => 'test',
      'title' => $title ?: "Link $nextId title",
      'langcode' => 'en',
      'link' => 'https://www.example.com/' . $this->randomString(),
      'description' => $description,
    ]));
    $this->assertEquals(SAVED_NEW, $link->save());
    $this->assertInstanceOf(MenuLinkContentInterface::class, $link);
    return $link;
  }

}
