<?php

declare(strict_types=1);

namespace Drupal\babel;

use Drupal\Component\Gettext\PoItem;
use Drupal\locale\LocaleLookup;

/**
 * Replacement for Locale's lookup class.
 *
 * @see https://www.drupal.org/project/drupal/issues/3533589
 */
class BabelLocaleLookup extends LocaleLookup {

  /**
   * Stores the Locale translation ID.
   *
   * NOTE: This is an addition compared to the original class.
   */
  protected ?string $sid = NULL;

  /**
   * {@inheritdoc}
   */
  public function resolveCacheMiss($offset) {
    $translation = $this->stringStorage->findTranslation([
      'language' => $this->langcode,
      'source' => $offset,
      'context' => $this->context,
    ]);

    if ($translation) {
      $value = !empty($translation->translation) ? $translation->translation : TRUE;
    }
    else {
      // We don't have the source string, update the {locales_source} table to
      // indicate the string is not translated.
      // NOTE: Next line is changed compared to the original class. We store the
      // newly created translation object.
      $translation = $this->stringStorage->createString([
        'source' => $offset,
        'context' => $this->context,
        'version' => \Drupal::VERSION,
      ])->addLocation('path', $this->requestStack->getCurrentRequest()->getRequestUri())->save();
      $value = TRUE;
    }

    // NOTE: Next line is added compared to the original class. We store the
    // newly created translation ID.
    $this->sid = (string) $translation->lid;

    // If there is no translation available for the current language then use
    // language fallback to try other translations.
    if ($value === TRUE) {
      $fallbacks = $this->languageManager->getFallbackCandidates([
        'langcode' => $this->langcode,
        'operation' => 'locale_lookup',
        'data' => $offset,
      ]);
      if (!empty($fallbacks)) {
        foreach ($fallbacks as $langcode) {
          $translation = $this->stringStorage->findTranslation([
            'language' => $langcode,
            'source' => $offset,
            'context' => $this->context,
          ]);

          if ($translation && !empty($translation->translation)) {
            $value = $translation->translation;
            break;
          }
        }
      }
    }

    if (is_string($value) && str_contains($value, PoItem::DELIMITER)) {
      // Community translations imported from localize.drupal.org as well as
      // migrated translations may contain @count[number].
      $value = preg_replace('!@count\[\d+\]!', '@count', $value);
    }

    $this->storage[$offset] = $value;
    // Disabling the usage of string caching allows a module to watch for
    // the exact list of strings used on a page. From a performance
    // perspective that is a terrible idea, so we have no user
    // interface for this. Be careful when turning this option off!
    if ($this->configFactory->get('locale.settings')->get('cache_strings')) {
      $this->persist($offset);
    }
    return $value;
  }

  /**
   * Returns the looked-up source ID.
   *
   * NOTE: This is an additional method compared to original class.
   *
   * @return string|null
   *   The source string ID.
   */
  public function getSourceId(): ?string {
    return $this?->sid;
  }

}
