<?php

declare(strict_types=1);

namespace Drupal\babel\Hook;

use Drupal\babel\BabelLocaleTranslation;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Cache\Context\BabelTranslateFormRoute;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Asset\AttachedAssetsInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Hooks for Babel module.
 */
class BabelHooks {

  public function __construct(
    protected LanguageManagerInterface $languageManager,
    protected Connection $database,
    protected BabelStorageInterface $babelStorage,
    #[Autowire(service: 'string_translator.locale.lookup')]
    protected BabelLocaleTranslation $babelTranslation,
    protected readonly AccountInterface $currentUser,
    #[Autowire(service: 'cache_context.babel_translate_form_route')]
    protected readonly BabelTranslateFormRoute $babelTranslateFormRoute,
  ) {}

  /**
   * Synchronizes tables {locales_source} and {babel_source_instance}.
   *
   * @param array $javascript
   *   An array of all JavaScript being presented on the page.
   * @param \Drupal\Core\Asset\AttachedAssetsInterface $assets
   *   The assets attached to the current response.
   * @param \Drupal\Core\Language\LanguageInterface $language
   *   The language for the page request that the assets will be rendered for.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('js_alter')]
  public function jsAlter(array &$javascript, AttachedAssetsInterface $assets, LanguageInterface $language): void {
    $jsStringIdsInLocaleQuery = $this->database->select('locales_location', 'll')
      ->distinct()
      ->fields('ll', ['sid'])
      ->condition('ll.type', 'javascript');
    $jsStringIdsInLocale = $jsStringIdsInLocaleQuery->execute()->fetchCol();

    $jsStringIdsInBabelQuery = clone $jsStringIdsInLocaleQuery;
    $jsStringIdsInBabelQuery->innerJoin('babel_source_instance', 'bsi', 'bsi.id = ll.sid');
    $jsStringIdsInBabel = $jsStringIdsInBabelQuery
      ->condition('bsi.plugin', 'locale')
      ->execute()
      ->fetchCol();

    $add = array_diff($jsStringIdsInLocale, $jsStringIdsInBabel);
    $delete = array_diff($jsStringIdsInBabel, $jsStringIdsInLocale);

    if ($add) {
      $this->babelTranslation->scheduleStringCopy(array_values($add));
    }

    if ($delete) {
      $this->babelStorage->scheduleDeleteById('locale', $delete);
    }
  }

  /**
   * Displays "Translate" tab in toolbar.
   *
   * @return array
   *   Array of tabs.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('toolbar')]
  public function toolbar(): array {
    $tab['babel_ui'] = [
      '#cache' => [
        'contexts' => [
          'user.permissions',
          'babel_translate_form_route',
        ],
      ],
    ];
    if (!$this->currentUser->hasPermission('translate interface')) {
      return $tab;
    }

    $routeParams = [];
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    if ($langcode !== 'en') {
      $routeParams['language'] = $langcode;
    }

    $tab['babel_ui'] += [
      '#type' => 'toolbar_item',
      'tab' => [
        '#type' => 'link',
        '#title' => t('Translate'),
        '#url' => Url::fromRoute('babel.ui', $routeParams)
          ->setOption('query', ['destination' => Url::fromRoute('<current>')->toString()]),
        '#attributes' => [
          'class' => ['use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => '80%',
            'classes' => [
              'ui-dialog' => 'ui-dialog--babel-translate',
            ],
          ]),
        ],
        '#access' => !$this->babelTranslateFormRoute->getContext(),
      ],
      '#wrapper_attributes' => [
        'class' => [
          'toolbar-tab--babel-translate',
        ],
      ],
    ];

    return $tab;
  }

  /**
   * Implements hook_theme().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'pager__babel' => [
        'base hook' => 'pager',
        'template' => 'pager',
        'file' => 'babel.theme.inc',
      ],
    ];
  }

}
