<?php

declare(strict_types=1);

namespace Drupal\babel\Controller;

use Drupal\babel\Ajax\PushParametersCommand;
use Drupal\babel\Form\BabelTranslateForm;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\ScrollTopCommand;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;
use Drupal\language\ConfigurableLanguageInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller of the route of the translation form's AJAX pager.
 */
class UiPager implements ContainerInjectionInterface {

  use AutowireTrait;

  public function __construct(
    protected readonly FormBuilderInterface $formBuilder,
  ) {}

  /**
   * Builds the response of the pager route.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param string $js
   *   Value of the path's {js} slug. This is used to determine whether the
   *   current request was an AJAX request.
   * @param \Drupal\language\ConfigurableLanguageInterface|null $language
   *   The language parameter, if any.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|\Symfony\Component\HttpFoundation\RedirectResponse
   *   AJAX response if the request was an AJAX request, or a redirect response.
   */
  public function updatePerPagination(
    Request $request,
    string $js,
    ?ConfigurableLanguageInterface $language,
  ): AjaxResponse|RedirectResponse {
    if ($js === 'ajax') {
      $response = new AjaxResponse();
      $form = $this->formBuilder->getForm(BabelTranslateForm::class, $language);
      $formSelector = 'form[data-babel-form-id="' . $form['#attributes']['data-babel-form-id'] . '"]';

      $params = [
        'langcode' => $request->query->get('langcode'),
        'translated' => $request->query->get('translated'),
        'search' => $request->query->get('search'),
        'include_inactive' => $request->query->get('include_inactive'),
        'page' => $request->query->get('page'),
      ];

      return $response
        ->addCommand(new PushParametersCommand($params))
        ->addCommand(new ReplaceCommand($formSelector, $form))
        ->addCommand(new ScrollTopCommand($formSelector));
    }

    $query = [
      'langcode' => $request->query->get('langcode'),
      'translated' => $request->query->get('translated'),
      'search' => $request->query->get('search'),
      'include_inactive' => $request->query->get('include_inactive'),
      'page' => $request->query->get('page'),
    ];
    if ($language) {
      unset($query['langcode']);
    }
    // Drop request destination if any.
    $request->query->remove('destination');
    return new RedirectResponse(
      Url::fromRoute(
        route_name: 'babel.ui',
        route_parameters: array_filter(['language' => $language?->getId()]),
        options: [
          'query' => $query,
        ],
    )->toString());
  }

}
