<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Kernel;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\EventSubscriber\ConfigSubscriber;
use Drupal\babel\Model\Source;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Tests\babel\Traits\ModuleInstallHelperTrait;

/**
 * Tests the status when creating new source string instances.
 *
 * @group babel
 */
class TranslationDefaultStatusTest extends BabelKernelTestBase {

  use ModuleInstallHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['babel_test'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->runInstallBatchOperations();

    // Create some locale source strings.
    $translation = $this->container->get('string_translator.locale.lookup');
    // Warm-up the \Drupal\locale\LocaleTranslation::$translations property.
    foreach (['foo', 'bar'] as $source) {
      $translation->getStringTranslation('pt', $source, '');
    }
    $translation->destruct();
  }

  /**
   * @covers \Drupal\babel\Plugin\Babel\TranslationType
   * @covers \Drupal\babel\BabelStorage
   */
  public function testDefaultStatus(): void {
    $manager = $this->container->get(TranslationTypePluginManager::class);
    $db = $this->container->get('database');
    $definitions = $manager->getDefinitions();
    $translation = $this->container->get('string_translator.locale.lookup');

    // Get all instances and group them by hash.
    $query = $db->select('babel_source_instance', 'bsi')
      ->fields('bsi', ['plugin'])
      ->fields('bs', ['status']);
    $query->innerJoin('babel_source', 'bs', 'bsi.hash = bs.hash');
    $instances = [];
    foreach ($query->execute() as $row) {
      $instances[$row->plugin][] = (bool) $row->status;
    }

    foreach (['locale' => FALSE, 'config' => TRUE] as $pluginId => $status) {
      // Check plugin definition alteration.
      $this->assertSame($status, $definitions[$pluginId]['status']);

      // Check statuses were created correctly in {babel_source} table.
      $expected = array_fill(0, count($instances[$pluginId]), $status);
      $this->assertSame($expected, $instances[$pluginId]);
    }

    // The 'system.site' config doesn't exist at this point.
    $config = $this->config('system.site');
    $this->assertTrue($config->isNew());
    // The 'foo' source string already exists as locale with status FALSE. Check
    // that the status is updated to TRUE, because the new instance is provided
    // by the 'config' plugin whose status defaults to TRUE.
    $config->set('name', 'foo')->save();
    $this->assertStatusPublished('foo');

    // Create a new config string.
    $config->set('slogan', 'the brown fox jumps over the lazy dog', '')->save();
    // Config strings are configured to be published by default.
    $this->assertStatusPublished('the brown fox jumps over the lazy dog');
    // Add a locale with the same string and context.
    $translation->getStringTranslation('pt', 'the brown fox jumps over the lazy dog', '');
    $translation->destruct();
    // Even locale strings are defaulting to FALSE, they cannot win over an
    // existing status TRUE.
    $this->assertStatusPublished('the brown fox jumps over the lazy dog');

    // Tests adding new source string with default status FALSE.
    $translation->getStringTranslation('pt', 'Jackdaws love my big sphinx of quartz', '');
    $translation->destruct();
    $this->assertStatusUnpublished('Jackdaws love my big sphinx of quartz');
  }

  /**
   * Asserts that as source string is published.
   *
   * @param string $sourceString
   *   The source string.
   * @param string $context
   *   (optional) The source string context. Defaults to empty string.
   */
  protected function assertStatusPublished(string $sourceString, string $context = ''): void {
    $this->statusAssertionHelper(TRUE, $sourceString, $context);
  }

  /**
   * Asserts that as source string is unpublished.
   *
   * @param string $sourceString
   *   The source string.
   * @param string $context
   *   (optional) The source string context. Defaults to empty string.
   */
  protected function assertStatusUnpublished(string $sourceString, string $context = ''): void {
    $this->statusAssertionHelper(FALSE, $sourceString, $context);
  }

  /**
   * Provides a helper for status assertion method.
   *
   * @param bool $expectedStatus
   *   The expected status.
   * @param string $sourceString
   *   The source string.
   * @param string $context
   *   (optional) The source string context. Defaults to empty string.
   */
  protected function statusAssertionHelper(bool $expectedStatus, string $sourceString, string $context): void {
    // Commit all changes.
    $this->container->get(ConfigSubscriber::class)->destruct();

    $source = new Source($sourceString, $context);
    $this->assertSame(
      $expectedStatus,
      $this->container->get(BabelStorageInterface::class)->getStatusForHash($source->getHash()),
    );
  }

}
