<?php

/**
 * @file
 * Theme related functions of Babel.
 */

declare(strict_types=1);

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\EventSubscriber\AjaxResponseSubscriber;
use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;

/**
 * Implements template_preprocess_HOOK() for pager__babel.
 *
 * Prepares variables for a Babel-specific AJAX pager.
 * Default template: pager.html.twig.
 *
 * @param array $variables
 *   An associative array containing:
 *   - pager: A render element containing:
 *   - #tags: An array of labels for the controls in the pager.
 *   - #element: An optional integer to distinguish between multiple pagers on
 *     one page.
 *   - #pagination_heading_level: An optional heading level for the pager.
 *   - #parameters: An associative array of query string parameters to append to
 *     the pager links.
 *   - #route_parameters: An associative array of the route parameters.
 *   - #quantity: The number of pages in the list.
 *
 * @see \template_preprocess_pager()
 */
function template_preprocess_pager__babel(array &$variables): void {
  if (
    empty($variables['pager']['#ajax_links']) ||
    empty($variables['items'])
  ) {
    return;
  }

  $variables['#attached']['library'][] = 'core/drupal.ajax';
  $routeName = $variables['pager']['#route_name'];
  $routeParams = $variables['pager']['#route_parameters'] ?? [];

  foreach ($variables['items']['pages'] as &$pageLinkItem) {
    _babel_process_babel_pager($pageLinkItem, $routeName, $routeParams);
  }

  foreach (['first', 'previous', 'next', 'last'] as $type) {
    if (!array_key_exists($type, $variables['items'])) {
      continue;
    }
    _babel_process_babel_pager($variables['items'][$type], $routeName, $routeParams);
  }
}

/**
 * Helper function to work around Drupal core issue #2925598.
 *
 * @param array $pager
 *   Array of the pager item with the following keys:
 *   - href: path of the pager item
 *   - attributes: template attribute object containing the HTML attributes of
 *     the pager item.
 * @param string $routeName
 *   Name of the target route of the pager.
 * @param array $routeParameters
 *   Parameters of the target route of the pager.
 *
 * @todo Revisit and / or remove when #2925598 gets resolved.
 * @see https://drupal.org/i/2925598
 */
function _babel_process_babel_pager(array &$pager, string $routeName, array $routeParameters): void {
  if (!empty($pager['href'])) {
    $parsed = UrlHelper::parse($pager['href']);
    $parsed['query'] = UrlHelper::filterQueryParameters(
      $parsed['query'],
      [
        FormBuilderInterface::AJAX_FORM_REQUEST,
        MainContentViewSubscriber::WRAPPER_FORMAT,
        AjaxResponseSubscriber::AJAX_REQUEST_PARAMETER,
        'ajax_page_state',
      ],
    );

    $pager['href'] = Url::fromRoute($routeName, $routeParameters, [
      'query' => $parsed['query'],
    ])->toString();
  }

  $pager['attributes']
    ->addClass('use-ajax')
    ->setAttribute('data-ajax-http-method', 'GET');
}
