/**
 * @file
 * Javascript functionality.
 */
((Drupal, once, $) => {
  Drupal.behaviors.babelTextareaResize = {
    attach(context) {
      once('babel-textarea-height-resize', 'html').forEach(() => {
        // React on window resize.
        window.addEventListener('resize', () => {
          Drupal.babelTextarea.recalculateAllTextareaHeight();
        });
        // React on Toolbar changes or to off canvas dialogs.
        // We us e jQuery because native addEventListener doesn't seem to work.
        // @todo Align when https://drupal.org/i/3446855 is resolved.
        $(document).on('drupalViewportOffsetChange', () => {
          Drupal.babelTextarea.recalculateAllTextareaHeight();
        });
      });

      once(
        'babel-textarea-height',
        Drupal.babelTextarea.selector,
        context,
      ).forEach((textarea) => {
        // CSS class is from resize.module.css in core.
        textarea.classList.remove(
          'resize-both',
          'resize-vertical',
          'resize-horizontal',
        );
        textarea.classList.add('resize-none');
        Drupal.babelTextarea.textAreaHeight(textarea);
        textarea.addEventListener('input', (event) => {
          Drupal.babelTextarea.textAreaHeight(event.target);
        });
      });
    },
  };

  /**
   * Adjusts the text area heights to fit the contained text.
   */
  Drupal.babelTextarea = {
    selector: 'textarea[data-babel-textarea-height]',
    recalculateAllTextareaHeight() {
      document
        .querySelectorAll(Drupal.babelTextarea.selector)
        .forEach((textarea) => {
          Drupal.babelTextarea.textAreaHeight(textarea);
        });
    },
    textAreaHeight(textarea) {
      // Default is 5 but show minimal for single-line strings.
      textarea.setAttribute('rows', 1);
      textarea.style.height = '';

      let height = textarea.scrollHeight;
      const groupId = textarea.dataset
        ? textarea.dataset.babelTextareaHeightGroup || null
        : null;
      const otherGroupMembers = [];
      if (groupId) {
        const groupMembers = document.querySelectorAll(
          `textarea[data-babel-textarea-height-group="${groupId}"]`,
        );
        [...groupMembers].forEach((item) => {
          if (item !== textarea) {
            otherGroupMembers.push(item);
          }
        });
      }

      otherGroupMembers.forEach((other) => {
        other.setAttribute('rows', 1);
        other.style.height = '';
        // For better vertical alignment the biggest height for both,
        // source and translation.
        height = Math.max(height, other.scrollHeight);

        other.style.height = `${height + 3}px`;
      });

      textarea.style.height = `${height + 3}px`;
    },
  };
})(Drupal, once, jQuery);
