<?php

declare(strict_types=1);

namespace Drupal\babel_menu_link_content;

use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\babel\Plugin\Babel\TranslationTypePluginManager;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\menu_link_content\MenuLinkContentInterface;

/**
 * Installation helper.
 */
class BabelMenuLinkContentService {

  /**
   * List of the menu link entity properties which aren't translated.
   *
   * @const string[]
   */
  private const EXCLUDED_FIELDS = [
    'content_translation_created',
    'content_translation_outdated',
    'content_translation_source',
    'content_translation_status',
    'content_translation_uid',
    'default_langcode',
    'langcode',
    'changed',
    'revision_translation_affected',
  ];

  /**
   * Static cache of System menu names.
   *
   * @var string[]
   */
  protected array $menuNames;

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly BabelStorageInterface $babelStorage,
    protected readonly LanguageManagerInterface $languageManager,
    protected readonly TranslationTypePluginManager $translationTypeManager,
  ) {}

  /**
   * Updates the source objects of the given menu link entity.
   *
   * @param \Drupal\menu_link_content\MenuLinkContentInterface $link
   *   The menu link content entity to process.
   */
  public function updateSourcesForLink(MenuLinkContentInterface $link): void {
    if ($this->isLinkAllowed($link)) {
      // Cleanup first.
      $this->babelStorage->delete('menu_link_content', $link->getMenuName() . ':' . $link->id() . ':');
      $this->babelStorage->update('menu_link_content', $this->getSourcesForLink($link));
    }
  }

  /**
   * Instantiates source objects from the properties of the given menu link.
   *
   * @param \Drupal\menu_link_content\MenuLinkContentInterface $link
   *   The menu link content entity to process.
   *
   * @return list<\Drupal\babel\Model\Source>
   *   List of source objects.
   */
  public function getSourcesForLink(MenuLinkContentInterface $link): array {
    $sources = [];

    if (!$this->isLinkAllowed($link)) {
      return $sources;
    }

    foreach ($link->getTranslatableFields(FALSE) as $field) {
      if (in_array($field->getName(), self::EXCLUDED_FIELDS, TRUE)) {
        continue;
      }

      foreach ($field as $delta => $item) {
        $id = $link->getMenuName() . ':' . $link->id() . ':' . $field->getName() . ':' . $delta;
        $sources[$id] = new Source(string: (string) $item->value, context: '');
      }
    }

    return $sources;
  }

  /**
   * Creates a batch process to process preexisting menu links entities.
   *
   * @param array $menuIds
   *   ID of the menus which menu link entities must be processed.
   */
  public function batchAddSources(array $menuIds = []): void {
    $menuLinkIdQuery = $this->entityTypeManager->getStorage('menu_link_content')
      ->getQuery()
      ->accessCheck(FALSE);

    $menuIds = $menuIds ? array_intersect($menuIds, $this->getMenuNames()) : $this->getMenuNames();
    if (!$menuIds) {
      return;
    }
    $menuLinkIdQuery->condition('menu_name', $menuIds, 'IN');

    $menuLinkIds = $menuLinkIdQuery->execute();
    if (!$menuLinkIds) {
      return;
    }
    $total = count($menuLinkIds);

    $batch = new BatchBuilder();
    foreach (array_chunk($menuLinkIds, 10) as $chunk) {
      $batch->addOperation([BabelMenuLinkContentBatchHelper::class, 'processMenuLinks'], [$chunk, $total]);
    }

    batch_set($batch->toArray());
  }

  /**
   * Returns a list of allowed menu names.
   *
   * @return string[]
   *   A list of allowed menu names.
   */
  protected function getMenuNames(): array {
    if (empty($this->menuNames)) {
      // Limit menus to the ones provided by System module (system.menu.*).
      $storage = $this->entityTypeManager->getStorage('menu');
      $allMenuNames = array_values($storage->getQuery()->execute());

      // Limit menus to the configured menu names.
      $plugin = $this->translationTypeManager->createInstance('menu_link_content');
      $configuredMenuNames = $plugin->getConfiguration()['menu'];
      $this->menuNames = $configuredMenuNames ? array_intersect($configuredMenuNames, $allMenuNames) : $allMenuNames;
    }
    return $this->menuNames;
  }

  /**
   * Check whether the passed link is handled by Babel.
   *
   * @param \Drupal\menu_link_content\MenuLinkContentInterface $link
   *   The menu link content entity.
   *
   * @return bool
   *   Whether the passed link is handled by Babel.
   */
  protected function isLinkAllowed(MenuLinkContentInterface $link): bool {
    $defaultLangcode = $this->languageManager->getDefaultLanguage()->getId();
    return in_array($link->getMenuName(), $this->getMenuNames(), TRUE) &&
      $link->language()->getId() === $defaultLangcode;
  }

}
