<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\Tests\babel\Traits\BabelFunctionalSetupTrait;
use Drupal\Tests\babel\Traits\TranslationAssertionTrait;

/**
 * Base class for Babel FunctionalJavascript tests.
 */
abstract class BabelFunctionalJavascriptTestBase extends WebDriverTestBase {

  use BabelFunctionalSetupTrait;
  use TranslationAssertionTrait;

  /**
   * Permissions required for operations.
   *
   * @const list<string>
   */
  protected const TEST_USER_PERMISSIONS = [
    'translate interface',
  ];

  /**
   * The ID of the language (language code) used for testing.
   *
   * @const string
   */
  protected const LANGCODE = 'pt-pt';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'babel',
    'block',
    'language',
    'config_translation',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->babelTestSetup();

    ConfigurableLanguage::createFromLangcode('de')->save();
  }

  /**
   * Tests the basic functionality of the Babel translate form.
   *
   * Verifies default filter values, functionality of the filters and also
   * confirms that translations are saved.
   */
  protected function assertTranslationForm(): void {
    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    $assertSession->fieldValueEquals('Language', 'de');
    // "Show inactive" must not be checked by default.
    $assertSession->checkboxNotChecked('Show inactive');
    $page->checkField('Show inactive');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    // "Show inactive" must be checked.
    $assertSession->checkboxChecked('Show inactive');
    $this->assertCount(
      50,
      $page->findAll('css', '.babel-list tbody > tr')
    );

    // Since we create the languages by the API, no translation import batches
    // are executed so there will be no translated strings.
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    $assertSession->fieldValueEquals('Type', 'translated');
    $assertSession->pageTextContains('0 items match the criteria');

    // Filtering for untranslated strings must report the same number as
    // filtering for any kind of strings.
    $page->selectFieldOption('Type', 'untranslated');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    $assertSession->fieldValueEquals('Type', 'untranslated');
    $this->assertTrue((bool) preg_match('#(\d+) items match the criteria#', $page->getText(), $matches));
    $untranslatedCount = $matches[1];

    $page->selectFieldOption('Type', 'all');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    $assertSession->fieldValueEquals('Type', 'all');
    $assertSession->pageTextContains(sprintf(
      '%s items match the criteria',
      $untranslatedCount,
    ));

    // Filter for sources or translations containing "administration".
    $page->fillField('Search', 'administration');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    // "Show inactive" must be still be checked, and we must see one result.
    $assertSession->checkboxChecked('Show inactive');
    $assertSession->pageTextContains('One item matches the criteria');
    $this->assertCount(
      1,
      $page->findAll('css', '.babel-list tbody > tr')
    );

    // Translate the "Administration" string to the test language.
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    $assertSession->fieldValueEquals('Language', static::LANGCODE);
    $this->translateSourceStringTo('Administration', '[TEST LANGCODE] Administration');
    $assertSession->statusMessageContainsAfterWait(
      'Translation of source string "Administration" in language Portuguese, Portugal was updated.',
      'status',
    );

    // Reset the UI and check how many translated string we have now.
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->assertWaitOnAjaxRequest();
    $assertSession->pageTextContains('One item matches the criteria');
    $assertSession->pageTextContains('Administration');
    $assertSession->pageTextContains('[TEST LANGCODE] Administration');
  }

}
