<?php

declare(strict_types=1);

namespace Drupal\babel\Plugin\Babel\DataTransfer;

use Drupal\Component\Utility\Variable;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\babel\Plugin\Babel\DataTransfer;
use Drupal\babel\Plugin\Babel\DataTransferPluginBase;
use Drupal\babel\Plugin\Babel\ExporterPluginInterface;
use Drupal\babel\Plugin\Babel\ImporterPluginInterface;
use Drupal\babel\Plugin\Babel\ImporterPluginTrait;
use Drupal\babel\Utility;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet as PhpOfficeSpreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Protection;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

/**
 * The spreadsheet data transfer plugin.
 */
#[DataTransfer(
  id: 'spreadsheet',
  label: new TranslatableMarkup('Spreadsheet'),
  fileExtensions: ['xlsx', 'xls', 'ods', 'csv'],
)]
class Spreadsheet extends DataTransferPluginBase implements ExporterPluginInterface, ImporterPluginInterface {

  use ImporterPluginTrait;

  private const HEADER = [
    'A1' => 'Source string (do not edit)',
    'B1' => 'Translated string',
    'C1' => 'Context (do not edit)',
    'D1' => 'Plural variant (do not edit)',
    'E1' => 'URL',
    'F1' => 'ID (do not edit)',
  ];

  private const PROTECTED = ['C', 'D', 'F', 'A1:F1'];

  private const EDITABLE = ['A', 'B', 'E'];

  private const WRITER = [
    'xls' => IOFactory::WRITER_XLS,
    'xlsx' => IOFactory::WRITER_XLSX,
    'ods' => IOFactory::WRITER_ODS,
    'csv' => IOFactory::WRITER_CSV,
  ];

  /**
   * {@inheritdoc}
   */
  public function createExportedFileContent(array $strings, string $langcode, string $extension): string {
    assert(isset(self::WRITER[$extension]));

    $languagePluralVariantsCount = (int) $this->getNumberOfPlurals($langcode);

    $spreadsheet = new PhpOfficeSpreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    foreach (self::HEADER as $cell => $value) {
      $sheet->setCellValue($cell, $value);
    }

    $row = 2;
    foreach ($strings as $hash => $string) {
      if (!$string->source->getStatus()) {
        continue;
      }

      $pluralVariantsCount = $string->isPlural() ? $languagePluralVariantsCount : 1;

      $sourceVariants = $string->getSourcePluralVariants();
      $translatedVariants = $string->getTranslatedPluralVariants();

      $sheet->setCellValue("C$row", $string->source->context);
      for ($index = 0; $index < $pluralVariantsCount; $index++) {
        $sheet
          ->setCellValue("A$row", $sourceVariants[$index] ?? '')
          ->setCellValue("B$row", $translatedVariants[$index] ?? '')
          ->setCellValue("D$row", $this->getPluralVariantLabel($pluralVariantsCount, $index))
          ->setCellValue("F$row", $hash);

        $sheet->getStyle("A$row")->getAlignment()->setWrapText(TRUE);
        $row++;
      }
    }

    // Cosmetics.
    $sheet->getStyle("A1:E1")->getAlignment()->setWrapText(TRUE);
    $sheet->getStyle("A1:E1")->getAlignment()->setVertical(Alignment::VERTICAL_TOP);
    $sheet->freezePane('A2');
    $sheet->getColumnDimensionByColumn(1)->setWidth(50);
    $sheet->getColumnDimensionByColumn(2)->setWidth(50);
    $sheet->getStyle("B")->getAlignment()->setWrapText(TRUE);
    $sheet->getColumnDimensionByColumn(3)->setWidth(15);
    $sheet->getStyle("C")->getAlignment()->setWrapText(TRUE);
    $sheet->getColumnDimensionByColumn(4)->setWidth(10);
    $sheet->getStyle("D")->getAlignment()->setWrapText(TRUE);
    $sheet->getColumnDimensionByColumn(5)->setWidth(40);
    $sheet->getColumnDimensionByColumn(6)->setCollapsed(TRUE);
    $sheet->getColumnDimensionByColumn(6)->setVisible(FALSE);

    // Protection.
    $sheet->getProtection()->setSheet(TRUE);
    $spreadsheet->getDefaultStyle()->getProtection()->setLocked(Protection::PROTECTION_UNPROTECTED);
    foreach (self::EDITABLE as $range) {
      $sheet->getStyle($range)->getProtection()->setLocked(Protection::PROTECTION_UNPROTECTED);
    }
    foreach (self::PROTECTED as $range) {
      $sheet->getStyle($range)->getProtection()->setLocked(Protection::PROTECTION_PROTECTED);
    }

    $writer = IOFactory::createWriter($spreadsheet, self::WRITER[$extension]);
    $file = $this->fileSystem->tempnam('temporary://', '');
    $writer->save($file);

    return file_get_contents($file);
  }

  /**
   * {@inheritdoc}
   */
  public function getImportedTranslations(string $path, string $langcode): array {
    $reader = IOFactory::createReaderForFile($path);
    $reader->setReadDataOnly(TRUE);
    $spreadsheet = $reader->load($path);
    $sheet = $spreadsheet->getActiveSheet();

    $this->validateSheet($sheet);

    $translations = [];
    foreach ($sheet->getRowIterator(2) as $row) {
      $iterator = $row->getCellIterator();
      $cell = $iterator->seek('F')->current();

      $hash = $cell->getValue();
      if ($warning = $this->hashIsValid($hash, $cell)) {
        $this->addImportWarning($warning);
        continue;
      }

      $cellValue = $iterator->seek('B')->current()?->getValue();
      $cellValue = is_string($cellValue) ? trim($cellValue) : $cellValue;
      $translations[$hash][] = (string) $cellValue;
    }

    return $translations;
  }

  /**
   * Validates the worksheet to be imported.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet instance.
   */
  protected function validateSheet(Worksheet $sheet): void {
    // Check header integrity.
    $header = $sheet->getRowIterator()->current();
    foreach ($header->getCellIterator() as $cell) {
      $cellValue = $cell->getValue();
      if ($cellValue !== self::HEADER[$cell->getCoordinate()]) {
        $this->addImportError($this->t('Invalid header at @coord: expected %expected, got %got.', [
          '@coord' => $cell->getCoordinate(),
          '%expected' => self::HEADER[$cell->getCoordinate()],
          '%got' => Variable::export($cellValue),
        ]));
      }
    }
  }

  /**
   * Validates a source string hash.
   *
   * @param string|int|null $hash
   *   The source string hash.
   * @param \PhpOffice\PhpSpreadsheet\Cell\Cell $cell
   *   The checked cell.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   NULL if there are no errors. Otherwise, an error message.
   */
  protected function hashIsValid(string|int|null $hash, Cell $cell): ?TranslatableMarkup {
    $args = ['%key' => (string) $hash, '@cell' => $cell->getCoordinate()];

    return match(TRUE) {
      empty($hash) => $this->t('The translation identifier at cell @cell is empty. Did you edit that column?', $args),
      !is_string($hash) || !preg_match('/^[a-z0-9]{64}$/', $hash) => t('The translation identifier at cell @cell is invalid. It should be a 64 characters alphanumeric string. Did you edit that column?', $args),
      !$this->babelStorage->hashExists($hash) => $this->t("The translation identifier %key at cell @cell is invalid or the source string don't exist anymore. Did you edit that column?", $args),
      default => NULL,
    };
  }

  /**
   * Returns the label of plural variant.
   *
   * @param int $pluralVariantsCount
   *   Number of plural variants.
   * @param int $pluralVariantIndex
   *   The index of the current plural variant.
   *
   * @return string
   *   A human-readable plural variant label.
   */
  protected function getPluralVariantLabel(int $pluralVariantsCount, int $pluralVariantIndex): string {
    return match(TRUE) {
      $pluralVariantsCount === 1 => '',
      $pluralVariantIndex === 0 => 'singular',
      $pluralVariantsCount === 2 && $pluralVariantIndex === 1 => 'plural',
      $pluralVariantsCount > 2 && $pluralVariantIndex === 1 => 'first plural',
      default => Utility::getOrdinal($pluralVariantIndex) . ' plural',
    };
  }

  /**
   * {@inheritdoc}
   */
  public function getExportGuidelines(): array {
    return [
      '#type' => 'markup',
      '#markup' => $this->t('XLS and XLSX format provide minor protection in relevant columns, rows and cells.'),
    ];
  }

}
