<?php

declare(strict_types=1);

namespace Drupal\Tests\babel\Functional;

use Drupal\babel\BabelConfigHelperInterface;
use Drupal\babel\BabelStorageInterface;
use Drupal\babel\Model\Source;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Url;

/**
 * Tests the translation UI without JavaScript interpreter.
 *
 * @group babel
 */
class TranslationInterfaceTest extends BabelFunctionalTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['babel_test_config'];

  /**
   * {@inheritdoc}
   */
  protected function babelTestSetup(): void {
    // Run test setup, but without creating any configurable language.
    $this->runPendingBatches();
    $this->placeDefaultBlocks();
    $this->testAccount = $this->drupalCreateUser(static::TEST_USER_PERMISSIONS);
  }

  /**
   * Tests access to translation UI with only default language, and filtering.
   */
  public function testOnlyDefaultLanguage(): void {
    $assertSession = $this->assertSession();
    $babelTranslateFormUrl = Url::fromRoute('babel.ui');
    $this->drupalGet($babelTranslateFormUrl);
    $assertSession->pageTextContains('You are not authorized to access this page.');

    $this->drupalLogin($this->testAccount);
    $this->drupalGet($babelTranslateFormUrl);

    $assertSession->pageTextContains('0 items match the criteria');
    // There wasn't any configurable language created so we must see the
    // "Add language" message.
    $assertSession->pageTextContains('No translatable languages available. Add a language first.');

    // Create a test language by API.
    $this->language = $this->createTestLanguage();

    // After creating a single language, we must be able to see strings, but not
    // the language selector.
    $this->drupalGet($babelTranslateFormUrl);
    // We must have more than 1 result.
    $assertSession->pageTextNotContains('0 items match the criteria');
    $assertSession->pageTextMatches('#\d+ items match the criteria#');
  }

  /**
   * Tests the translation UI without JavaScript.
   *
   * @depends testOnlyDefaultLanguage
   */
  public function testFilterAndTranslation(): void {
    // Before Drupal 11.2, the configs in the babel_test_config module
    // are not loaded (this happens only after changes in
    // #3416522. Because of that and only for needed versions, we do here
    // the work that is automatic after Drupal 11.2.
    // @todo Remove after compatibility with Drupal < 11.1 is dropped.
    if (floatval(\Drupal::VERSION) < 11.2) {
      $name = 'babel_test_config';
      $data = \Drupal::service('config.storage')->read("$name.content");
      $paths = \Drupal::service(BabelConfigHelperInterface::class)->getTranslatableProperties("$name.content");
      $sources = [];
      foreach ($paths as $path => $translationContext) {
        $sources["$name.content:$path"] = new Source(
          string: NestedArray::getValue($data, explode('.', $path)),
          context: $translationContext,
        );
      }
      \Drupal::service(BabelStorageInterface::class)->update('config', $sources);
    }

    $this->createTestLanguage(static::LANGCODE);
    $this->createTestLanguage('de');
    $page = $this->getSession()->getPage();
    $assertSession = $this->assertSession();

    $this->drupalLogin($this->testAccount);
    $this->drupalGet($babelTranslateUrl = Url::fromRoute('babel.ui'));

    $assertSession->fieldValueEquals('Language', 'de');
    // "Show inactive" must not be checked by default.
    $assertSession->checkboxNotChecked('Show inactive');
    $page->checkField('Show inactive');
    $page->pressButton('Filter');
    // "Show inactive" must be checked.
    $assertSession->checkboxChecked('Show inactive');
    $this->assertCount(
      50,
      $page->findAll('css', '.babel-list tbody > tr')
    );

    // Since we create the languages by the API, no translation import batches
    // are executed so there will be no translated strings.
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->fieldValueEquals('Type', 'translated');
    $assertSession->pageTextContains('0 items match the criteria');

    // Filtering for untranslated strings must report the same number as
    // filtering for any kind of strings.
    $page->selectFieldOption('Type', 'untranslated');
    $page->pressButton('Filter');
    $assertSession->fieldValueEquals('Type', 'untranslated');
    $this->assertTrue((bool) preg_match('#(\d+) items match the criteria#', $page->getText(), $matches));
    $untranslatedCount = $matches[1];

    $page->selectFieldOption('Type', 'all');
    $page->pressButton('Filter');
    $assertSession->fieldValueEquals('Type', 'all');
    $assertSession->pageTextContains(sprintf(
      '%s items match the criteria',
      $untranslatedCount,
    ));

    // Filter for sources or translations containing "administration".
    $page->fillField('Contains', 'administration');
    $page->pressButton('Filter');
    // "Show inactive" must be still be checked, and we must see one result.
    $assertSession->checkboxChecked('Show inactive');
    $assertSession->pageTextContains('One item matches the criteria');
    $this->assertCount(
      1,
      $page->findAll('css', '.babel-list tbody > tr')
    );

    // Translate the "Administration" string to the test language.
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->pressButton('Filter');
    $assertSession->fieldValueEquals('Language', static::LANGCODE);
    $this->translateSourceStringTo('Administration', '[TEST LANGCODE] Administration');
    $assertSession->pageTextContains('Translation of source string "Administration" in language Portuguese, Portugal was updated.');

    // Reset the UI and check how many translated string we have now.
    $this->drupalGet($babelTranslateUrl);
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->pageTextContains('One item matches the criteria');
    $assertSession->pageTextContains('Administration');
    $assertSession->pageTextContains('[TEST LANGCODE] Administration');

    // Test removing translation.
    $this->translateSourceStringTo('Administration', '');
    $assertSession->pageTextContains('Translation of source string "Administration" in language Portuguese, Portugal was deleted.');

    // Test translating string with plural.
    $page->fillField('Contains', '1 hello');
    $page->selectFieldOption('Type', 'all');
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->pressButton('Filter');
    $this->translateSourceStringTo('1 hello', '[TEST LANGCODE] 1 hello', 0);
    $this->translateSourceStringTo('@count hellos', '[TEST LANGCODE] @count hellos', 1);
    $assertSession->pageTextContains('Translations of source strings "1 hello, @count hellos" in language Portuguese, Portugal were updated.');
    // Reset the UI and check how many translated string we have now.
    $this->drupalGet($babelTranslateUrl);
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->pageTextContains('One item matches the criteria');
    $assertSession->pageTextContains('[TEST LANGCODE] 1 hello');
    $assertSession->pageTextContains('[TEST LANGCODE] @count hellos');
    // Test removing the singular translation.
    $translation_cell = $page->find('css', 'tr.babel-translate-row > td[data-babel-translation]');
    $translation_hash = $translation_cell->getAttribute('data-babel-translation');
    $page->fillField("edit-list-{$translation_hash}-translation-" . static::LANGCODE . '-0', '');
    $page->pressButton("edit-list-{$translation_hash}-save");
    $assertSession->pageTextContains('Translations of source strings "1 hello, @count hellos" in language Portuguese, Portugal were updated.');
    $assertSession->fieldValueEquals('Singular form', '', $translation_cell);
    $assertSession->fieldValueEquals('Plural form', '[TEST LANGCODE] @count hellos', $translation_cell);
    // Reset the UI and check if the translation is correct.
    $this->drupalGet($babelTranslateUrl);
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->selectFieldOption('Type', 'translated');
    $page->pressButton('Filter');
    $assertSession->pageTextContains('One item matches the criteria');
    $assertSession->fieldValueEquals('Singular form', '', $translation_cell);
    $assertSession->fieldValueEquals('Plural form', '[TEST LANGCODE] @count hellos', $translation_cell);
    // Test removing the plural translation.
    $translation_cell = $page->find('css', 'tr.babel-translate-row > td[data-babel-translation]');
    $translation_hash = $translation_cell->getAttribute('data-babel-translation');
    $page->fillField("edit-list-{$translation_hash}-translation-" . static::LANGCODE . '-1', '');
    $page->pressButton("edit-list-{$translation_hash}-save");
    $assertSession->pageTextContains('Translations of source strings "1 hello, @count hellos" in language Portuguese, Portugal were deleted.');
    // Reset the UI and check if the translation is correct.
    $this->drupalGet($babelTranslateUrl);
    $page->fillField('Contains', '1 hello');
    $page->selectFieldOption('Language', static::LANGCODE);
    $page->selectFieldOption('Type', 'all');
    $page->pressButton('Filter');
    $assertSession->pageTextContains('One item matches the criteria');
    $assertSession->fieldValueEquals('Singular form', '', $translation_cell);
    $assertSession->fieldValueEquals('Plural form', '', $translation_cell);
  }

}
